<?php
// Common label settings + ZPL helpers for SM Produce

require_once __DIR__ . '/db_remote.php';

/**
 * Ensure label_settings table exists and return current row (id=1) or defaults.
 */
function get_label_settings(mysqli $mysqli): array
{
    $mysqli->query("
        CREATE TABLE IF NOT EXISTS label_settings (
            id INT PRIMARY KEY,
            printer_name VARCHAR(255) DEFAULT NULL,
            layout       VARCHAR(10)  NOT NULL DEFAULT 'L1',
            auto_print_mode ENUM('OFF','ASK','AUTO') NOT NULL DEFAULT 'ASK',
            fields_json  JSON DEFAULT NULL,
            updated_at   TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4
    ");

    $res = $mysqli->query("SELECT * FROM label_settings WHERE id = 1 LIMIT 1");
    if ($res && $row = $res->fetch_assoc()) {
        $row['fields'] = [];
        if (!empty($row['fields_json'])) {
            $decoded = json_decode($row['fields_json'], true);
            if (is_array($decoded)) {
                $row['fields'] = $decoded;
            }
        }
        return $row;
    }

    // default fields
    $fields = [
        'grower'     => true,
        'variety'    => true,
        'type'       => false,
        'lot'        => true,
        'date'       => true,
        'barcode'    => true,
        'big_grower' => true,
        'border'     => true,
    ];
    $json = json_encode($fields, JSON_UNESCAPED_UNICODE);

    $stmt = $mysqli->prepare("
        INSERT INTO label_settings (id, printer_name, layout, auto_print_mode, fields_json)
        VALUES (1, NULL, 'L1', 'ASK', ?)
        ON DUPLICATE KEY UPDATE fields_json = VALUES(fields_json)
    ");
    if ($stmt) {
        $stmt->bind_param("s", $json);
        $stmt->execute();
    }

    return [
        'id'             => 1,
        'printer_name'   => null,
        'layout'         => 'L1',
        'auto_print_mode'=> 'ASK',
        'fields_json'    => $json,
        'fields'         => $fields,
    ];
}

/**
 * Build ZPL for a single FULL bin label.
 * Label size target: 4x2" (approx 812x406 dots at 203dpi).
 */
function build_full_bin_zpl(array $bin, array $settings): string
{
    $fields  = $settings['fields'] ?? [];
    $layout  = $settings['layout'] ?? 'L1';

    $grower  = $bin['grower']  ?? '';
    $variety = $bin['variety'] ?? '';
    $type    = $bin['type']    ?? '';
    $lot     = $bin['lot']     ?? '';
    $date    = $bin['date']    ?? '';
    $id      = (int)($bin['id'] ?? 0);
    $barcode = "FBIN-" . str_pad((string)$id, 6, '0', STR_PAD_LEFT);

    // Base header
    $zpl  = "^XA\n";
    $zpl .= "^PW812\n";   // 4" width
    $zpl .= "^LL406\n";   // 2" height
    $zpl .= "^LH0,0\n";   // label home

    // Optional border
    if (!empty($fields['border'])) {
        $zpl .= "^FO10,10^GB792,386,2^FS\n";
    }

    $y = 30;

    // Big grower on top
    if (!empty($fields['big_grower'])) {
        $zpl .= "^FO40,$y^A0N,80,80^FD" . zpl_escape($grower) . "^FS\n";
        $y += 90;
    } elseif (!empty($fields['grower'])) {
        $zpl .= "^FO40,$y^A0N,40,40^FD" . zpl_escape($grower) . "^FS\n";
        $y += 50;
    }

    // Variety + Type line
    if (!empty($fields['variety']) || !empty($fields['type'])) {
        $line = [];
        if (!empty($fields['variety']) && $variety !== '') {
            $line[] = $variety;
        }
        if (!empty($fields['type']) && $type !== '') {
            $line[] = $type;
        }
        if (!empty($line)) {
            $zpl .= "^FO40,$y^A0N,35,35^FD" . zpl_escape(implode(' - ', $line)) . "^FS\n";
            $y += 45;
        }
    }

    // Lot + Date line
    if (!empty($fields['lot']) || !empty($fields['date'])) {
        $parts = [];
        if (!empty($fields['lot']) && $lot !== '') {
            $parts[] = "Lot: " . $lot;
        }
        if (!empty($fields['date']) && $date !== '') {
            $parts[] = "Date: " . $date;
        }
        if (!empty($parts)) {
            $zpl .= "^FO40,$y^A0N,30,30^FD" . zpl_escape(implode("   ", $parts)) . "^FS\n";
            $y += 40;
        }
    }

    // Barcode (CODE128)
    if (!empty($fields['barcode'])) {
        $zpl .= "^BY3,2,80\n";
        $zpl .= "^FO100,220^BCN,80,Y,N,N\n";
        $zpl .= "^FD" . zpl_escape($barcode) . "^FS\n";
        // Human readable below
        $zpl .= "^FO100,310^A0N,30,30^FD" . zpl_escape($barcode) . "^FS\n";
    }

    $zpl .= "^XZ\n";
    return $zpl;
}

/**
 * Escape ZPL field data.
 */
function zpl_escape(string $text): string
{
    // Replace ^ and ~ which are ZPL control chars
    $text = str_replace(['^', '~'], ' ', $text);
    return $text;
}

/**
 * Send ZPL to Zebra Print Agent (or similar) running on localhost.
 * Adjust URL/path according to your agent configuration.
 */
function send_zpl_to_printer(string $zpl, ?string $printerName = null): array
{
    $url = "http://localhost:9100/print";

    if ($printerName) {
        $url .= "?printer=" . urlencode($printerName);
    }

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_POST           => true,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER     => ['Content-Type: text/plain'],
        CURLOPT_POSTFIELDS     => $zpl,
        CURLOPT_TIMEOUT        => 10,
    ]);

    $body   = curl_exec($ch);
    $errNo  = curl_errno($ch);
    $errStr = curl_error($ch);
    $code   = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    return [
        'http_code' => $code,
        'error_no'  => $errNo,
        'error'     => $errStr,
        'body'      => $body,
    ];
}
