<?php
/**
 * barcode_alert_checker.php
 *
 * Script da eseguire in background (es. ogni minuto) per controllare
 * l'inattività del lettore Keyence e, dopo X minuti di nessuna lettura,
 * inviare UN SOLO report di produzione completo.
 *
 * Il timeout X (in minuti) è configurabile da:
 *   Settings → Email & Timeout Alerts → Timeout Alerts
 */

require_once __DIR__ . '/config/db_remote.php';
require_once __DIR__ . '/config/emailer.php';
require_once __DIR__ . '/send_report.php';

if (!($mysqli instanceof mysqli)) {
    error_log('barcode_alert_checker: $mysqli non è una connessione valida.');
    exit;
}

// Carichiamo configurazione alert (timeout, stato)
$alerts = sp_email_get_alerts();

// Se gli alert non sono abilitati, non facciamo nulla
if (empty($alerts['enabled'])) {
    // alert disabilitato da Settings
    exit;
}

// Timeout configurato (minuti)
$timeoutMinutes = (int)($alerts['timeout_minutes'] ?? 60);
if ($timeoutMinutes <= 0) {
    $timeoutMinutes = 60;
}

// Recuperiamo l'ultima lettura dal lettore (ultima riga in barcode_readings)
$sql = "SELECT MAX(read_time) AS last_read FROM barcode_readings";
$res = $mysqli->query($sql);

if (!$res) {
    error_log('barcode_alert_checker: query fallita: ' . $mysqli->error);
    exit;
}

$row = $res->fetch_assoc();
$res->free();

$lastRead = $row['last_read'] ?? null;

// Se non è mai stato letto nulla, non facciamo nulla
if (!$lastRead) {
    exit;
}

$lastReadTs = strtotime($lastRead);
if ($lastReadTs === false) {
    error_log('barcode_alert_checker: strtotime fallito su ' . var_export($lastRead, true));
    exit;
}

$nowTs = time();
$idleMinutes = (int)floor(($nowTs - $lastReadTs) / 60);

// Se il tempo di inattività è inferiore al timeout, non inviamo nulla
if ($idleMinutes < $timeoutMinutes) {
    // Nessuna azione, ma NON resettiamo ancora lo stato: ci basta il controllo per timestamp
    exit;
}

// Controlliamo se abbiamo già inviato un report per QUESTO stesso timestamp di ultima lettura
// Salviamo in JSON un campo 'last_auto_scan_time' con l'ultima read_time per cui è stato inviato il report
$lastAutoScanTime = $alerts['last_auto_scan_time'] ?? null;

if ($lastAutoScanTime && $lastAutoScanTime === $lastRead) {
    // Abbiamo già inviato un report per questo periodo di inattività → non inviamo di nuovo
    exit;
}

// Se arriviamo qui:
// - il lettore è inattivo da almeno timeoutMinutes
// - NON abbiamo ancora inviato il report per questo valore di last_read_time

$ok = sp_send_production_report(true); // true = report completo

if ($ok) {
    // Salviamo il timestamp dell'ultima scan per cui abbiamo mandato report automatico
    $alerts['last_auto_scan_time'] = $lastRead;
    sp_email_save_alerts($alerts);
} else {
    error_log('barcode_alert_checker: sp_send_production_report() failed.');
}

exit;
