<?php
require_once __DIR__ . '/../config/user_functions.php';

if (session_status() === PHP_SESSION_NONE) { session_start(); }
require_once __DIR__ . '/../config/users_local.php';
require_once __DIR__ . '/../config/emailer.php';
require_once __DIR__ . '/../includes/db.php';

if (!user_has_permission('settings')) {
    http_response_code(403);
    include '../includes/header.php';
    include '../includes/sidebar.php';
    echo "<div class='container-fluid py-4'><h3 class='text-danger'>Access denied</h3></div>";
    include '../includes/footer.php';
    exit;
}

$emailCfg        = sp_email_get_settings();
$emailRecipients = ['to' => sp_email_get_recipients()];
$alertsCfg       = sp_email_get_alerts();

$emailFlash  = '';
$recipFlash  = '';
$alertsFlash = '';

/* ---------------------------
   helpers
--------------------------- */
function sp_bool($v): int { return !empty($v) ? 1 : 0; }

function sp_table_has_column(PDO $pdo, string $table, string $col): bool {
    try {
        $stmt = $pdo->prepare("
            SELECT COUNT(*) 
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = DATABASE()
              AND TABLE_NAME = ?
              AND COLUMN_NAME = ?
        ");
        $stmt->execute([$table, $col]);
        return (int)$stmt->fetchColumn() > 0;
    } catch (Throwable $e) {
        return false;
    }
}

$hasRoleCases   = sp_table_has_column($pdo, 'settings_sources', 'role_cases');
$hasRoleDumping = sp_table_has_column($pdo, 'settings_sources', 'role_dumping');
$hasRolePallet  = sp_table_has_column($pdo, 'settings_sources', 'role_pallet');

/* ---------------------------
   AJAX: scanner test endpoint
--------------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['scanner_test'])) {
    $barcode = trim((string)($_POST['test_barcode'] ?? ''));
    $selectedScannerCode = trim((string)($_POST['test_scanner_code'] ?? ''));
    $expectedUse = trim((string)($_POST['expected_use'] ?? 'cases')); // cases|dumping|pallet

    if ($expectedUse !== 'cases' && $expectedUse !== 'dumping' && $expectedUse !== 'pallet') {
        $expectedUse = 'cases';
    }

    $detected = [
        'source' => 'unknown',        // keyence|symcode|manual|unknown
        'scanner_code' => $selectedScannerCode ?: 'unknown',
        'scanner_label' => $selectedScannerCode ?: 'Unknown',
        'roles' => [],
        'status' => 'ignored',        // accepted|blocked|ignored
        'reason' => '',
    ];

    $now = time();

    // detect Keyence if last keyence scan matches AND is recent (<= 5s)
    try {
        $row = $pdo->query("SELECT code, received_at FROM keyence_scans ORDER BY id DESC LIMIT 1")->fetch(PDO::FETCH_ASSOC);
        if ($row && $barcode !== '') {
            $lastCode = (string)$row['code'];
            $lastTs = strtotime((string)$row['received_at']);
            if ($lastTs !== false && ($now - $lastTs) <= 5 && strcasecmp($lastCode, $barcode) === 0) {
                $detected['source'] = 'keyence';
                $detected['scanner_code'] = 'keyence';
            }
        }
    } catch (Throwable $e) {}

    if ($detected['source'] === 'unknown') {
        if ($barcode === '') {
            $detected['source'] = 'manual';
        } else {
            $detected['source'] = 'symcode';
        }
    }

    // Load scanner config for selected scanner code (or detected keyence)
    $cfg = null;
    try {
        $codeToLoad = $detected['scanner_code'];
        if ($codeToLoad === 'unknown' || $codeToLoad === '') {
            $codeToLoad = $selectedScannerCode ?: 'unknown';
        }
        if ($detected['source'] === 'keyence') {
            $codeToLoad = 'keyence';
        }

        $st = $pdo->prepare("SELECT * FROM settings_sources WHERE code = ? LIMIT 1");
        $st->execute([$codeToLoad]);
        $cfg = $st->fetch(PDO::FETCH_ASSOC) ?: null;
    } catch (Throwable $e) {
        $cfg = null;
    }

    if ($cfg) {
        $detected['scanner_label'] = (string)($cfg['label'] ?? $detected['scanner_label']);

        $roles = [];
        if ($hasRoleCases && !empty($cfg['role_cases']))   { $roles[] = 'cases'; }
        if ($hasRoleDumping && !empty($cfg['role_dumping'])) { $roles[] = 'dumping'; }
        if ($hasRolePallet && !empty($cfg['role_pallet']))  { $roles[] = 'pallet'; }

        // Backward compat: if only is_production exists, treat as cases
        if (!$hasRoleCases && !$hasRoleDumping && !$hasRolePallet) {
            if (!empty($cfg['is_production'])) { $roles[] = 'cases'; }
        } else {
            // If roles columns exist but are all 0, allow legacy is_production to imply cases
            if (empty($roles) && !empty($cfg['is_production'])) { $roles[] = 'cases'; }
        }

        $detected['roles'] = $roles;

        // Decide accept/block for expected use
        if (!$selectedScannerCode && $detected['source'] === 'symcode') {
            // symcode keyboard cannot be auto-identified: require operator selection
            $detected['status'] = 'blocked';
            $detected['reason'] = 'Select which Symcode you are testing (keyboard scanners are not auto-identifiable).';
        } else {
            if (in_array($expectedUse, $roles, true)) {
                $detected['status'] = 'accepted';
                $detected['reason'] = 'OK';
            } else {
                $detected['status'] = 'blocked';
                $detected['reason'] = 'Scanner not enabled for this function.';
            }
        }
    } else {
        $detected['status'] = 'blocked';
        $detected['reason'] = 'Scanner not found in settings_sources.';
    }

    // Audit log (best effort)
    try {
        $username = $_SESSION['username'] ?? $_SESSION['user'] ?? '';
        $ip = $_SERVER['REMOTE_ADDR'] ?? '';
        $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $rolesCsv = is_array($detected['roles']) ? implode(',', $detected['roles']) : '';

        $pdo->prepare("
            INSERT INTO scanner_audit_log
            (barcode, detected_source, scanner_code, scanner_label, expected_use, roles, status, reason, username, ip_address, user_agent)
            VALUES (?,?,?,?,?,?,?,?,?,?,?)
        ")->execute([
            $barcode,
            $detected['source'],
            $detected['scanner_code'],
            $detected['scanner_label'],
            $expectedUse,
            $rolesCsv,
            $detected['status'],
            $detected['reason'],
            (string)$username,
            (string)$ip,
            (string)$ua
        ]);
    } catch (Throwable $e) {}

    header('Content-Type: application/json; charset=utf-8');
    echo json_encode([
        'barcode' => $barcode,
        'source' => $detected['source'],
        'scanner_code' => $detected['scanner_code'],
        'scanner_label' => $detected['scanner_label'],
        'expected_use' => $expectedUse,
        'roles' => $detected['roles'],
        'status' => $detected['status'],
        'reason' => $detected['reason'],
        'ts' => date('Y-m-d H:i:s')
    ], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    exit;
}

/* ---------------------------
   HANDLE POST: SMTP / RECIPIENTS / ALERTS
--------------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    if (isset($_POST['email_action'])) {
        $action = $_POST['email_action'];

        if ($action === 'save_smtp' || $action === 'test_smtp') {
            $emailCfg['smtp_host'] = trim($_POST['smtp_host'] ?? '');
            $emailCfg['smtp_port'] = (int)($_POST['smtp_port'] ?? 587);
            if ($emailCfg['smtp_port'] <= 0) { $emailCfg['smtp_port'] = 587; }

            $emailCfg['smtp_user'] = trim($_POST['smtp_user'] ?? '');
            $smtpPass = $_POST['smtp_pass'] ?? '';
            if ($smtpPass !== '') { $emailCfg['smtp_pass'] = $smtpPass; }

            $secure = strtolower(trim($_POST['smtp_secure'] ?? 'tls'));
            if (!in_array($secure, ['tls', 'ssl', 'none'], true)) { $secure = 'tls'; }
            $emailCfg['smtp_secure'] = $secure;

            $emailCfg['from_email'] = trim($_POST['from_email'] ?? '');
            if ($emailCfg['from_email'] === '') { $emailCfg['from_email'] = 'itsmproduce@gmail.com'; }

            $emailCfg['from_name'] = trim($_POST['from_name'] ?? '');
            if ($emailCfg['from_name'] === '') { $emailCfg['from_name'] = 'SM Produce Reports'; }

            sp_email_save_settings($emailCfg);
            $emailFlash = 'SMTP settings saved.';

            if ($action === 'test_smtp') {
                $testTo = trim($_POST['test_email_to'] ?? $emailCfg['from_email']);
                if ($testTo !== '') {
                    $subject = 'Test email from SM Produce';
                    $body    = "This is a test email from SM Produce settings.<br><br>If you received this, Gmail SMTP is working.";
                    $ok = sp_smtp_send($subject, $body, [$testTo]);
                    $emailFlash = $ok
                        ? "Test email sent via Gmail SMTP to {$testTo}."
                        : "Failed to send test email via Gmail SMTP. Check credentials and Gmail app password.";
                } else {
                    $emailFlash = 'Test email address is empty.';
                }
            }
        }

        if ($action === 'add_recipient') {
            $new = trim($_POST['new_recipient'] ?? '');
            if ($new !== '' && filter_var($new, FILTER_VALIDATE_EMAIL)) {
                if (!in_array($new, $emailRecipients['to'], true)) {
                    $emailRecipients['to'][] = $new;
                    sp_email_set_recipients($emailRecipients['to']);
                    $recipFlash = "Recipient {$new} added.";
                } else {
                    $recipFlash = "Recipient {$new} already exists.";
                }
            } else {
                $recipFlash = 'Invalid email address.';
            }
        }

        if ($action === 'remove_recipient') {
            $rem = trim($_POST['recipient_email'] ?? '');
            if ($rem !== '') {
                $emailRecipients['to'] = array_values(array_filter(
                    $emailRecipients['to'],
                    static fn($e) => $e !== $rem
                ));
                sp_email_set_recipients($emailRecipients['to']);
                $recipFlash = "Recipient {$rem} removed.";
            }
        }
    }

    if (isset($_POST['alerts_action']) && $_POST['alerts_action'] === 'save_alerts') {
        $alertsCfg['enabled'] = !empty($_POST['alerts_enabled']);
        $tm = (int)($_POST['timeout_minutes'] ?? 60);
        if ($tm <= 0) { $tm = 60; }
        $alertsCfg['timeout_minutes'] = $tm;
        sp_email_save_alerts($alertsCfg);
        $alertsFlash = 'Timeout alert settings saved.';
    }
}

/* ---------------------------
   SCANNER SOURCES SAVE (your module, extended)
--------------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_sources'])) {
    foreach (($_POST['label'] ?? []) as $code => $label) {
        $code = (string)$code;
        $label = (string)$label;

        $isProd = isset($_POST['is_production'][$code]) ? 1 : 0;

        $roleCases   = isset($_POST['role_cases'][$code]) ? 1 : 0;
        $roleDumping = isset($_POST['role_dumping'][$code]) ? 1 : 0;
        $rolePallet  = isset($_POST['role_pallet'][$code]) ? 1 : 0;

        if ($hasRoleCases || $hasRoleDumping || $hasRolePallet) {
            $stmt = $pdo->prepare("
                UPDATE settings_sources
                SET label = ?, is_production = ?, role_cases = ?, role_dumping = ?, role_pallet = ?
                WHERE code = ?
            ");
            $stmt->execute([$label, $isProd, $roleCases, $roleDumping, $rolePallet, $code]);
        } else {
            $stmt = $pdo->prepare("UPDATE settings_sources SET label = ?, is_production = ? WHERE code = ?");
            $stmt->execute([$label, $isProd, $code]);
        }
    }
    header('Location: settings.php?sources_saved=1');
    exit;
}

try {
    $sources = $pdo->query("SELECT * FROM settings_sources ORDER BY code")->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
    $sources = [];
}

include '../includes/header.php';
include '../includes/sidebar.php';
?>
<div class="container-fluid py-4">
  <h3 class="mb-4">Settings</h3>
  <div class="row g-4">

    <div class="col-md-4">
      <div class="card h-100">
        <div class="card-body">
          <h5 class="card-title">Printers</h5>
          <p class="card-text">Configure default printers and printing behavior.</p>
          <a href="/smproduce/pages/settings_printers.php" class="btn btn-outline-primary">Printer Settings</a>
        </div>
      </div>
    </div>

    <div class="col-md-4">
      <div class="card h-100">
        <div class="card-body">
          <h5 class="card-title">Full Bin Labels</h5>
          <p class="card-text">Choose which fields appear on full bin labels.</p>
          <a href="/smproduce/pages/settings_labels.php" class="btn btn-outline-primary">Label Options</a>
        </div>
      </div>
    </div>

    <div class="col-md-4">
      <div class="card h-100">
        <div class="card-body">
          <h5 class="card-title">Reports</h5>
          <p class="card-text">Select default printer and which columns to include in reports.</p>
          <a href="/smproduce/pages/settings_reports.php" class="btn btn-outline-primary">Report Options</a>
        </div>
      </div>
    </div>

    <div class="col-md-4">
      <div class="card h-100">
        <div class="card-body">
          <h5 class="card-title">Orders Printing</h5>
          <p class="card-text">Choose printer and auto-print options for orders.</p>
          <a href="/smproduce/pages/settings_orders.php" class="btn btn-outline-primary">Orders Printer</a>
        </div>
      </div>
    </div>

    <div class="col-md-4">
      <div class="card h-100 border-info">
        <div class="card-body">
          <h5 class="card-title">Production Settings</h5>
          <p class="card-text">Configure production summary fields, charts and refresh options.</p>
          <a href="/smproduce/pages/settings_production.php" class="btn btn-outline-info">
            Production Settings
          </a>
        </div>
      </div>
    </div>

  </div>

  <div class="card mt-4 border-info">
    <div class="card-header bg-info text-white"><strong>Scanner Sources</strong></div>
    <div class="card-body">
      <?php if (isset($_GET['sources_saved'])): ?>
        <div class="alert alert-success py-2">Scanner sources saved successfully.</div>
      <?php endif; ?>

      <?php if (empty($sources)): ?>
        <div class="alert alert-warning">settings_sources table not found or empty.</div>
      <?php else: ?>
      <form method="post">
        <input type="hidden" name="save_sources" value="1">

        <table class="table table-sm align-middle">
          <thead>
            <tr>
              <th>Code</th>
              <th>Name</th>
              <th class="text-center">Production</th>
              <th class="text-center">Cases</th>
              <th class="text-center">Dumping</th>
              <th class="text-center">Pallet</th>
            </tr>
          </thead>
          <tbody>
          <?php foreach ($sources as $s): $c = (string)$s['code']; ?>
            <tr>
              <td><code><?= htmlspecialchars($c) ?></code></td>
              <td>
                <input type="text"
                       name="label[<?= htmlspecialchars($c, ENT_QUOTES) ?>]"
                       value="<?= htmlspecialchars((string)($s['label'] ?? '')) ?>"
                       class="form-control form-control-sm">
              </td>

              <td class="text-center">
                <input type="checkbox"
                       name="is_production[<?= htmlspecialchars($c, ENT_QUOTES) ?>]"
                       <?= !empty($s['is_production']) ? 'checked' : '' ?>>
              </td>

              <td class="text-center">
                <input type="checkbox"
                       name="role_cases[<?= htmlspecialchars($c, ENT_QUOTES) ?>]"
                       <?= !empty($s['role_cases']) ? 'checked' : '' ?>
                       <?= $hasRoleCases ? '' : 'disabled' ?>>
              </td>

              <td class="text-center">
                <input type="checkbox"
                       name="role_dumping[<?= htmlspecialchars($c, ENT_QUOTES) ?>]"
                       <?= !empty($s['role_dumping']) ? 'checked' : '' ?>
                       <?= $hasRoleDumping ? '' : 'disabled' ?>>
              </td>

              <td class="text-center">
                <input type="checkbox"
                       name="role_pallet[<?= htmlspecialchars($c, ENT_QUOTES) ?>]"
                       <?= !empty($s['role_pallet']) ? 'checked' : '' ?>
                       <?= $hasRolePallet ? '' : 'disabled' ?>>
              </td>
            </tr>
          <?php endforeach; ?>
          </tbody>
        </table>

        <button class="btn btn-primary btn-sm">Save Scanner Settings</button>
      </form>
      <?php endif; ?>
    </div>
  </div>

  <div class="card mt-4 border-warning">
    <div class="card-header bg-warning"><strong>Scanner Test + Simulator</strong></div>
    <div class="card-body">
      <form id="scannerTestForm">
        <input type="hidden" name="scanner_test" value="1">

        <div class="row g-2 align-items-end">
          <div class="col-md-4">
            <label class="form-label">Which Symcode are you testing?</label>
            <select class="form-select" name="test_scanner_code" id="test_scanner_code">
              <option value="">(Select)</option>
              <?php foreach ($sources as $s): ?>
                <?php if ((string)$s['code'] === 'keyence') continue; ?>
                <option value="<?= htmlspecialchars((string)$s['code'], ENT_QUOTES) ?>">
                  <?= htmlspecialchars((string)$s['label'] . " (" . (string)$s['code'] . ")") ?>
                </option>
              <?php endforeach; ?>
              <option value="keyence">Keyence (keyence)</option>
            </select>
          </div>

          <div class="col-md-4">
            <label class="form-label">Expected use</label>
            <select class="form-select" name="expected_use" id="expected_use">
              <option value="cases">Cases count</option>
              <option value="dumping">Dumping bins</option>
              <option value="pallet">Palletization</option>
            </select>
          </div>

          <div class="col-md-4">
            <label class="form-label">Scan / type barcode</label>
            <input type="text" class="form-control form-control-lg"
                   name="test_barcode" id="scannerTestInput"
                   placeholder="Scan here..." autocomplete="off">
          </div>
        </div>
      </form>

      <div class="mt-3" id="scannerFeedback"></div>
    </div>
  </div>

  <div class="mt-4">
    <?php if ($emailFlash): ?><div class="alert alert-info py-2 mb-2"><?= htmlspecialchars($emailFlash) ?></div><?php endif; ?>
    <?php if ($recipFlash): ?><div class="alert alert-info py-2 mb-2"><?= htmlspecialchars($recipFlash) ?></div><?php endif; ?>
    <?php if ($alertsFlash): ?><div class="alert alert-info py-2 mb-2"><?= htmlspecialchars($alertsFlash) ?></div><?php endif; ?>
  </div>

  <div class="row mt-2 g-4">
    <div class="col-lg-8">
      <div class="card shadow-sm">
        <div class="card-header bg-light">
          <h5 class="mb-0">Email & Timeout Alerts</h5>
          <small class="text-muted">Gmail SMTP settings, recipients and Keyence inactivity timeout.</small>
        </div>
        <div class="card-body">

          <h6 class="mb-3">SMTP Settings (Gmail)</h6>
          <form method="post" class="mb-4">
            <input type="hidden" name="email_action" value="save_smtp">

            <div class="row g-3">
              <div class="col-md-6">
                <label class="form-label">SMTP Host</label>
                <input type="text" name="smtp_host" class="form-control form-control-sm"
                       value="<?= htmlspecialchars($emailCfg['smtp_host'] ?? '') ?>">
              </div>
              <div class="col-md-3">
                <label class="form-label">SMTP Port</label>
                <input type="number" name="smtp_port" class="form-control form-control-sm"
                       value="<?= htmlspecialchars((string)($emailCfg['smtp_port'] ?? 587)) ?>">
              </div>
              <div class="col-md-3">
                <label class="form-label">Encryption</label>
                <select name="smtp_secure" class="form-select form-select-sm">
                  <?php $sec = strtolower($emailCfg['smtp_secure'] ?? 'tls'); ?>
                  <option value="tls"  <?= $sec === 'tls'  ? 'selected' : '' ?>>TLS</option>
                  <option value="ssl"  <?= $sec === 'ssl'  ? 'selected' : '' ?>>SSL</option>
                  <option value="none" <?= $sec === 'none' ? 'selected' : '' ?>>None</option>
                </select>
              </div>
            </div>

            <div class="row g-3 mt-2">
              <div class="col-md-6">
                <label class="form-label">SMTP User (Gmail address)</label>
                <input type="text" name="smtp_user" class="form-control form-control-sm"
                       value="<?= htmlspecialchars($emailCfg['smtp_user'] ?? '') ?>">
              </div>
              <div class="col-md-6">
                <label class="form-label">SMTP Password (App Password)</label>
                <input type="password" name="smtp_pass" class="form-control form-control-sm"
                       placeholder="●●●●●●●● (leave blank to keep current)">
              </div>
            </div>

            <div class="row g-3 mt-2">
              <div class="col-md-6">
                <label class="form-label">From email</label>
                <input type="email" name="from_email" class="form-control form-control-sm"
                       value="<?= htmlspecialchars($emailCfg['from_email'] ?? '') ?>">
              </div>
              <div class="col-md-6">
                <label class="form-label">From name</label>
                <input type="text" name="from_name" class="form-control form-control-sm"
                       value="<?= htmlspecialchars($emailCfg['from_name'] ?? '') ?>">
              </div>
            </div>

            <div class="row g-3 mt-3 align-items-end">
              <div class="col-md-6">
                <label class="form-label">Test email to</label>
                <input type="email" name="test_email_to" class="form-control form-control-sm"
                       placeholder="you@example.com">
              </div>
              <div class="col-md-6 text-end">
                <button type="submit" class="btn btn-primary btn-sm me-2">Save SMTP Settings</button>
                <button type="submit" name="email_action" value="test_smtp" class="btn btn-outline-secondary btn-sm">Send Test Email</button>
              </div>
            </div>
          </form>

          <hr>

          <h6 class="mb-3">Email Recipients</h6>

          <form method="post" class="row g-2 mb-3">
            <input type="hidden" name="email_action" value="add_recipient">
            <div class="col-md-8">
              <input type="email" name="new_recipient" class="form-control form-control-sm"
                     placeholder="recipient@example.com" required>
            </div>
            <div class="col-md-4 d-grid d-md-flex justify-content-md-end">
              <button type="submit" class="btn btn-success btn-sm">Add recipient</button>
            </div>
          </form>

          <?php if (empty($emailRecipients['to'])): ?>
            <div class="text-muted small mb-3">No recipients configured yet.</div>
          <?php else: ?>
            <div class="table-responsive mb-3">
              <table class="table table-sm align-middle mb-0">
                <thead>
                  <tr>
                    <th style="width:60%">Email</th>
                    <th style="width:40%" class="text-end">Actions</th>
                  </tr>
                </thead>
                <tbody>
                  <?php foreach ($emailRecipients['to'] as $mail): ?>
                    <tr>
                      <td><?= htmlspecialchars($mail) ?></td>
                      <td class="text-end">
                        <form method="post" class="d-inline">
                          <input type="hidden" name="email_action" value="remove_recipient">
                          <input type="hidden" name="recipient_email" value="<?= htmlspecialchars($mail) ?>">
                          <button type="submit" class="btn btn-outline-danger btn-sm">Remove</button>
                        </form>
                      </td>
                    </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          <?php endif; ?>

          <hr>

          <h6 class="mb-3">Timeout Alerts (Keyence barcode inactivity)</h6>
          <form method="post">
            <input type="hidden" name="alerts_action" value="save_alerts">

            <div class="row g-3 align-items-center">
              <div class="col-md-4">
                <div class="form-check">
                  <input class="form-check-input" type="checkbox" id="alerts_enabled"
                         name="alerts_enabled" value="1" <?= !empty($alertsCfg['enabled']) ? 'checked' : '' ?>>
                  <label class="form-check-label" for="alerts_enabled">Enable inactivity reports</label>
                </div>
              </div>
              <div class="col-md-4">
                <label class="form-label mb-0">Timeout (minutes)</label>
                <input type="number" min="1" max="1440"
                       name="timeout_minutes"
                       class="form-control form-control-sm"
                       value="<?= htmlspecialchars((string)($alertsCfg['timeout_minutes'] ?? 60)) ?>">
              </div>
              <div class="col-md-4 text-end mt-3 mt-md-0">
                <button type="submit" class="btn btn-primary btn-sm">Save alert settings</button>
              </div>
            </div>
          </form>

        </div>
      </div>
    </div>
  </div>

</div>

<?php
if (!function_exists('sp_cmd_run')) {
    function sp_cmd_run($cmd) {
        $out = @shell_exec($cmd . ' 2>&1');
        if ($out === null) $out = '';
        return trim($out);
    }
}

$keyence_service = 'keyence_listener';
$cloud_service   = 'Cloudflared';

$keyence_raw = sp_cmd_run('sc query ' . $keyence_service);
$keyence_running = stripos($keyence_raw, 'RUNNING') !== false;
$keyence_stopped = stripos($keyence_raw, 'STOPPED') !== false;

$cloud_raw = sp_cmd_run('sc query ' . $cloud_service);
$cloud_running = stripos($cloud_raw, 'RUNNING') !== false;
$cloud_stopped = stripos($cloud_raw, 'STOPPED') !== false;

$keyence_ip   = '192.168.1.180';
$keyence_port = 8600;

$keyence_console = '';
$cloud_console   = '';

$ka = $_POST['keyence_action'] ?? $_GET['keyence_action'] ?? null;
$ca = $_POST['cloud_action']   ?? $_GET['cloud_action']   ?? null;

if ($ka) {
    if ($ka === 'start') { $keyence_console = sp_cmd_run('nssm start ' . $keyence_service); }
    elseif ($ka === 'stop') { $keyence_console = sp_cmd_run('nssm stop ' . $keyence_service); }
    elseif ($ka === 'restart') { $keyence_console = sp_cmd_run('nssm restart ' . $keyence_service); }
    elseif ($ka === 'ping') { $keyence_console = sp_cmd_run('ping -n 2 ' . escapeshellarg($keyence_ip)); }
    elseif ($ka === 'telnet') {
        $errno = 0; $errstr = '';
        $t0 = microtime(true);
        $fp = @fsockopen($keyence_ip, $keyence_port, $errno, $errstr, 2);
        $dt = (microtime(true) - $t0) * 1000;
        if ($fp) { fclose($fp); $keyence_console = "TCP OK to {$keyence_ip}:{$keyence_port} ({$dt} ms)"; }
        else { $keyence_console = "TCP FAIL to {$keyence_ip}:{$keyence_port} — {$errstr} ({$errno})"; }
    }
}

if ($ca) {
    if ($ca === 'start') { $cloud_console = sp_cmd_run('sc start ' . $cloud_service); }
    elseif ($ca === 'stop') { $cloud_console = sp_cmd_run('sc stop ' . $cloud_service); }
    elseif ($ca === 'restart') {
        $cloud_console  = sp_cmd_run('sc stop ' . $cloud_service);
        $cloud_console .= "\n" . sp_cmd_run('sc start ' . $cloud_service);
    } elseif ($ca === 'status') { $cloud_console = sp_cmd_run('sc query ' . $cloud_service); }
}

$keyence_ping_ok = false;
$ping_out = sp_cmd_run('ping -n 1 ' . escapeshellarg($keyence_ip));
if ($ping_out && stripos($ping_out, 'TTL=') !== false) { $keyence_ping_ok = true; }

$keyence_port_ok = false;
$errno = 0; $errstr = '';
$fp = @fsockopen($keyence_ip, $keyence_port, $errno, $errstr, 1);
if ($fp) { $keyence_port_ok = true; fclose($fp); }

$last_msg = '(no data yet)';
$last_path = __DIR__ . '/../keyence_listener/last_keyence_message.txt';
if (is_file($last_path)) { $last_msg = htmlspecialchars(trim(file_get_contents($last_path))); }
?>

<div class="container-fluid mt-4" style="max-width:1100px;">

  <div class="card mb-4 shadow-sm border-primary mx-auto" id="keyence-section">
    <div class="card-header bg-primary text-white">
      <strong>Keyence Settings</strong>
    </div>
    <div class="card-body">

      <div class="row mb-3 text-center">
        <div class="col-md-4 mb-2">
          <h6 class="mb-1">Listener service</h6>
          <?php if ($keyence_running): ?>
            <span class="badge bg-success px-3 py-2">Running</span>
          <?php elseif ($keyence_stopped): ?>
            <span class="badge bg-danger px-3 py-2">Stopped</span>
          <?php else: ?>
            <span class="badge bg-secondary px-3 py-2">Unknown</span>
          <?php endif; ?>
          <div class="small text-muted mt-1"><code><?= htmlspecialchars($keyence_service) ?></code></div>
        </div>

        <div class="col-md-4 mb-2">
          <h6 class="mb-1">Ping <?= htmlspecialchars($keyence_ip) ?></h6>
          <?php if ($keyence_ping_ok): ?>
            <span class="badge bg-success px-3 py-2">Reachable</span>
          <?php else: ?>
            <span class="badge bg-danger px-3 py-2">No reply</span>
          <?php endif; ?>
        </div>

        <div class="col-md-4 mb-2">
          <h6 class="mb-1">Port <?= (int)$keyence_port ?></h6>
          <?php if ($keyence_port_ok): ?>
            <span class="badge bg-success px-3 py-2">OPEN</span>
          <?php else: ?>
            <span class="badge bg-danger px-3 py-2">CLOSED</span>
          <?php endif; ?>
        </div>
      </div>

      <form method="post" class="mb-3 d-flex flex-wrap gap-2">
        <button name="keyence_action" value="start" class="btn btn-success btn-sm">Start listener</button>
        <button name="keyence_action" value="stop" class="btn btn-danger btn-sm">Stop listener</button>
        <button name="keyence_action" value="restart" class="btn btn-warning btn-sm">Restart listener</button>
        <button name="keyence_action" value="ping" class="btn btn-outline-primary btn-sm">Ping Keyence</button>
        <button name="keyence_action" value="telnet" class="btn btn-outline-secondary btn-sm">Test port 8600</button>
        <a href="keyence_monitor.php" class="btn btn-outline-dark btn-sm">Monitor</a>
        <a href="keyence_logs.php" class="btn btn-outline-dark btn-sm">Logs</a>
        <a href="keyence_restart.php" class="btn btn-outline-dark btn-sm">Restart page</a>
      </form>

      <label class="form-label fw-semibold">Keyence console output</label>
      <pre class="bg-dark text-light p-3 rounded" style="min-height:90px; white-space:pre-wrap;"><?=
        $keyence_console ? htmlspecialchars($keyence_console) : 'Ready.'
      ?></pre>

      <label class="form-label fw-semibold mt-3">Last Keyence message</label>
      <pre class="bg-light p-3 rounded" style="min-height:70px;"><?= $last_msg ?></pre>

    </div>
  </div>

  <div class="card mb-5 shadow-sm border-success mx-auto" id="cloudflare-section">
    <div class="card-header bg-success text-white">
      <strong>Cloudflare Tunnel</strong>
    </div>
    <div class="card-body text-center">

      <div class="mb-3">
        <h6 class="mb-1">Cloudflared service</h6>
        <?php if ($cloud_running): ?>
          <span class="badge bg-success px-3 py-2">Active</span>
        <?php elseif ($cloud_stopped): ?>
          <span class="badge bg-danger px-3 py-2">Stopped</span>
        <?php else: ?>
          <span class="badge bg-secondary px-3 py-2">Unknown</span>
        <?php endif; ?>
        <div class="small text-muted mt-1"><code><?= htmlspecialchars($cloud_service) ?></code></div>
      </div>

      <form method="post" class="mb-3 d-flex flex-wrap gap-2 justify-content-center">
        <button name="cloud_action" value="start" class="btn btn-success btn-sm">Start tunnel</button>
        <button name="cloud_action" value="stop" class="btn btn-danger btn-sm">Stop tunnel</button>
        <button name="cloud_action" value="restart" class="btn btn-warning btn-sm">Restart tunnel</button>
        <button name="cloud_action" value="status" class="btn btn-outline-primary btn-sm">Status</button>
      </form>

      <label class="form-label fw-semibold">Cloudflared console output</label>
      <pre class="bg-dark text-light p-3 rounded" style="min-height:90px; white-space:pre-wrap;"><?=
        $cloud_console ? htmlspecialchars($cloud_console) : 'Ready.'
      ?></pre>

    </div>
  </div>

</div>

<?php include '../includes/footer.php'; ?>

<script>
(function(){
  const input = document.getElementById('scannerTestInput');
  const feedback = document.getElementById('scannerFeedback');
  const form = document.getElementById('scannerTestForm');
  if (!input || !feedback || !form) return;

  function beep(kind, roles) {
    const AudioContext_ = window.AudioContext || window.webkitAudioContext;
    if (!AudioContext_) return;
    const ctx = new AudioContext_();
    const o = ctx.createOscillator();
    const g = ctx.createGain();
    o.connect(g); g.connect(ctx.destination);

    let freq = 500;
    if (kind === 'ok') freq = 880;
    if (kind === 'warn') freq = 520;
    if (kind === 'err') freq = 220;

    // role-based tweak
    if (Array.isArray(roles) && roles.includes('dumping')) freq += 60;
    if (Array.isArray(roles) && roles.includes('pallet')) freq += 120;

    o.frequency.value = freq;
    g.gain.value = 0.08;
    o.start();
    setTimeout(()=>{ o.stop(); ctx.close(); }, 150);
  }

  function badge(status) {
    if (status === 'accepted') return { cls:'alert-success', label:'ACCEPTED', sound:'ok' };
    if (status === 'blocked') return { cls:'alert-danger', label:'BLOCKED', sound:'err' };
    return { cls:'alert-warning', label:'IGNORED', sound:'warn' };
  }

  input.addEventListener('keydown', function(e){
    if (e.key !== 'Enter') return;
    e.preventDefault();

    const fd = new FormData(form);
    fetch('', { method:'POST', body: fd })
      .then(r => r.json())
      .then(d => {
        const b = badge(d.status);
        beep(b.sound, d.roles);

        const rolesText = (Array.isArray(d.roles) && d.roles.length) ? d.roles.join(', ') : '-';

        feedback.innerHTML = `
          <div class="alert ${b.cls}">
            <div class="d-flex justify-content-between align-items-center">
              <div><strong>${b.label}</strong></div>
              <div class="small text-muted">${d.ts || ''}</div>
            </div>
            <div class="mt-2">
              <div><strong>Barcode:</strong> ${d.barcode || ''}</div>
              <div><strong>Detected source:</strong> ${d.source || ''}</div>
              <div><strong>Scanner:</strong> ${d.scanner_label || ''} <span class="text-muted">(${d.scanner_code || ''})</span></div>
              <div><strong>Expected use:</strong> ${d.expected_use || ''}</div>
              <div><strong>Enabled roles:</strong> ${rolesText}</div>
              <div><strong>Reason:</strong> ${d.reason || ''}</div>
            </div>
          </div>
        `;
        input.value = '';
      })
      .catch(() => {
        feedback.innerHTML = `<div class="alert alert-danger">Test failed.</div>`;
      });
  });
})();
</script>
