<?php
require_once __DIR__ . '/../config/user_functions.php';
if (session_status() === PHP_SESSION_NONE) { session_start(); }
require_once __DIR__ . '/../config/users_local.php';
require_once __DIR__ . '/../config/db_remote.php';

if (function_exists('user_has_permission') && !user_has_permission('reports')) {
    http_response_code(403);
    include '../includes/header.php';
    include '../includes/sidebar.php';
    echo "<div class='container-fluid py-4'><h3 class='text-danger'>Access denied</h3></div>";
    include '../includes/footer.php';
    exit;
}

$settingsFile = __DIR__ . '/../config/production_settings.json';
$cfg = file_exists($settingsFile) ? (json_decode(file_get_contents($settingsFile), true) ?: []) : [];
$refreshInterval = max(2, (int)($cfg['refresh_interval'] ?? 5));

$range = $_GET['range'] ?? 'day';
if (!in_array($range, ['day','week','month','all'], true)) $range = 'day';

$from = isset($_GET['from']) ? trim((string)$_GET['from']) : '';
$to   = isset($_GET['to'])   ? trim((string)$_GET['to'])   : '';

$useCustom = false;
if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $from) && preg_match('/^\d{4}-\d{2}-\d{2}$/', $to)) {
    $useCustom = true;
    if (strcmp($from, $to) > 0) { $tmp = $from; $from = $to; $to = $tmp; }
    $whereDate = "s.scanned_at >= '{$mysqli->real_escape_string($from)} 00:00:00' AND s.scanned_at <= '{$mysqli->real_escape_string($to)} 23:59:59'";
} else {
    switch ($range) {
        case 'week':  $whereDate = "s.scanned_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)"; break;
        case 'month': $whereDate = "s.scanned_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)"; break;
        case 'all':   $whereDate = "1=1"; break;
        default:      $whereDate = "DATE(s.scanned_at) = CURDATE()"; break;
    }
}

$rangeLabel = $useCustom ? ("{$from} → {$to}") : ([
    'day' => 'Today',
    'week' => 'Last 7 days',
    'month' => 'Last 30 days',
    'all' => 'All time',
][$range] ?? 'Today');

if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=production_summary.csv');
    $out = fopen('php://output', 'w');
    fputcsv($out, ['time','barcode','source','scan_function','scanner_id','SKU','variety','packaging','size','grower']);

    $sql = "
        SELECT
          s.scanned_at,
          s.code,
          s.source,
          s.scan_function,          c.SKU,
          c.variety,
          c.packaging,
          c.size,
          c.grower
        FROM barcode_scans s
        LEFT JOIN casecodes c ON c.serial = s.code
        WHERE {$whereDate}
        ORDER BY s.scanned_at DESC
    ";
    if ($res = $mysqli->query($sql)) {
        while ($r = $res->fetch_assoc()) {
            fputcsv($out, $r);
        }
        $res->free();
    }
    fclose($out);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['manual_barcode'])) {
    $code = trim((string)$_POST['manual_barcode']);
    if ($code !== '') {
        $stmt = $mysqli->prepare("INSERT IGNORE INTO barcode_scans (code, source, scan_function) VALUES (?, 'manual', 'case_count')");
        if ($stmt) {
            $stmt->bind_param("s", $code);
            $stmt->execute();
            $stmt->close();
        }
    }
    $qs = "range=" . urlencode($range);
    if ($useCustom) { $qs .= "&from=" . urlencode($from) . "&to=" . urlencode($to); }
    header("Location: production_summary.php?{$qs}");
    exit;
}

$groupMode = 'date';
if (!$useCustom && $range === 'day') $groupMode = 'hour';
if ($useCustom && $from === $to) $groupMode = 'hour';

$kpi = ['keyence'=>0,'symcode'=>0,'manual'=>0,'total'=>0];
$sql = "
    SELECT s.source, COUNT(*) AS c
    FROM barcode_scans s
    WHERE {$whereDate} AND s.scan_function='case_count'
    GROUP BY s.source
";
if ($res = $mysqli->query($sql)) {
    while ($r = $res->fetch_assoc()) {
        $src = $r['source'];
        $c = (int)$r['c'];
        if (isset($kpi[$src])) $kpi[$src] = $c;
        $kpi['total'] += $c;
    }
    $res->free();
}

$stackLabels = [];
$stack = [];
if ($groupMode === 'hour') {
    $sql = "
        SELECT DATE_FORMAT(s.scanned_at, '%Y-%m-%d %H:00') AS b, s.source, COUNT(*) AS c
        FROM barcode_scans s
        WHERE {$whereDate} AND s.scan_function='case_count'
        GROUP BY b, s.source
        ORDER BY b
    ";
} else {
    $sql = "
        SELECT DATE(s.scanned_at) AS b, s.source, COUNT(*) AS c
        FROM barcode_scans s
        WHERE {$whereDate} AND s.scan_function='case_count'
        GROUP BY b, s.source
        ORDER BY b
    ";
}
if ($res = $mysqli->query($sql)) {
    while ($r = $res->fetch_assoc()) {
        $b = (string)$r['b'];
        if (!isset($stack[$b])) $stack[$b] = ['keyence'=>0,'symcode'=>0,'manual'=>0];
        $src = $r['source'];
        if (isset($stack[$b][$src])) $stack[$b][$src] = (int)$r['c'];
    }
    $res->free();
}
$stackLabels = array_keys($stack);
$stackKeyence = [];
$stackSymcode = [];
$stackManual  = [];
foreach ($stackLabels as $lbl) {
    $stackKeyence[] = (int)($stack[$lbl]['keyence'] ?? 0);
    $stackSymcode[] = (int)($stack[$lbl]['symcode'] ?? 0);
    $stackManual[]  = (int)($stack[$lbl]['manual'] ?? 0);
}

$topSku = [];
$skuSource = [];
$sql = "
    SELECT
      COALESCE(c.SKU,'(n/a)') AS SKU,
      COALESCE(NULLIF(TRIM(c.variety),''),'(n/a)') AS variety,
      COALESCE(NULLIF(TRIM(c.packaging),''),'(n/a)') AS packaging,
      COALESCE(NULLIF(TRIM(c.size),''),'(n/a)') AS size,
      s.source,
      COUNT(*) AS c
    FROM barcode_scans s
    LEFT JOIN casecodes c ON c.serial = s.code
    WHERE {$whereDate} AND s.scan_function='case_count'
    GROUP BY SKU, variety, packaging, size, s.source
";
if ($res = $mysqli->query($sql)) {
    $tmp = [];
    while ($r = $res->fetch_assoc()) {
        $sku = (string)$r['SKU'];
        $desc = trim($r['variety'].' - '.$r['packaging'].' - '.$r['size'], ' -');
        if ($desc === '') $desc = '(n/a)';
        $src = (string)$r['source'];
        $cnt = (int)$r['c'];
        $key = $sku . '||' . $desc;

        if (!isset($tmp[$key])) {
            $tmp[$key] = ['SKU'=>$sku,'description'=>$desc,'keyence'=>0,'symcode'=>0,'manual'=>0,'total'=>0];
        }
        if (isset($tmp[$key][$src])) $tmp[$key][$src] += $cnt;
        $tmp[$key]['total'] += $cnt;

        if (!isset($skuSource[$sku])) $skuSource[$sku] = ['keyence'=>0,'symcode'=>0,'manual'=>0,'total'=>0];
        if (isset($skuSource[$sku][$src])) $skuSource[$sku][$src] += $cnt;
        $skuSource[$sku]['total'] += $cnt;
    }
    $res->free();

    $topSku = array_values($tmp);
    usort($topSku, function($a,$b){ return $b['total'] <=> $a['total']; });
    $topSku = array_slice($topSku, 0, 50);
}

$sourceTopSku = ['keyence'=>[],'symcode'=>[],'manual'=>[]];
$sql = "
    SELECT s.source, COALESCE(c.SKU,'(n/a)') AS SKU, COUNT(*) AS c
    FROM barcode_scans s
    LEFT JOIN casecodes c ON c.serial = s.code
    WHERE {$whereDate} AND s.scan_function='case_count'
    GROUP BY s.source, SKU
    ORDER BY s.source, c DESC
";
if ($res = $mysqli->query($sql)) {
    while ($r = $res->fetch_assoc()) {
        $src = (string)$r['source'];
        if (!isset($sourceTopSku[$src])) continue;
        if (count($sourceTopSku[$src]) >= 15) continue;
        $sourceTopSku[$src][] = ['SKU'=>(string)$r['SKU'],'count'=>(int)$r['c']];
    }
    $res->free();
}

$live = [];
$sql = "
    SELECT
      s.id,
      s.scanned_at,
      s.code,
      s.source,
      s.scan_function,      c.SKU,
      c.variety,
      c.packaging,
      c.size,
      c.grower
    FROM barcode_scans s
    LEFT JOIN casecodes c ON c.serial = s.code
    WHERE {$whereDate} AND s.scan_function='case_count'
    ORDER BY s.scanned_at DESC
    LIMIT 250
";
if ($res = $mysqli->query($sql)) {
    while ($r = $res->fetch_assoc()) $live[] = $r;
    $res->free();
}

$stopMinutes = 10;
$lastScanTs = null;
$sql = "SELECT MAX(s.scanned_at) AS last_ts FROM barcode_scans s WHERE {$whereDate} AND s.scan_function='case_count'";
if ($res = $mysqli->query($sql)) {
    $row = $res->fetch_assoc();
    if (!empty($row['last_ts'])) $lastScanTs = strtotime($row['last_ts']);
    $res->free();
}
$lineStopped = false;
$ageSeconds = null;
if ($lastScanTs) {
    $ageSeconds = time() - $lastScanTs;
    if ($ageSeconds > ($stopMinutes * 60)) $lineStopped = true;
}

$readers = [];
if ($res = $mysqli->query("SELECT code, label FROM settings_sources ORDER BY code")) {
    while ($r = $res->fetch_assoc()) {
        $readers[] = ['code'=>$r['code'],'label'=>$r['label']];
    }
    $res->free();
}

include '../includes/header.php';
include '../includes/sidebar.php';
?>
<div class="container-fluid py-4">

  <div class="d-flex justify-content-between align-items-center flex-wrap gap-2 mb-3">
    <div>
      <h3 class="mb-0">Production Summary</h3>
      <div class="text-muted small">Range: <?= htmlspecialchars($rangeLabel) ?></div>
    </div>

    <div class="d-flex gap-2 align-items-center flex-wrap">
      <a class="btn btn-outline-secondary btn-sm" href="/smproduce/pages/settings_production.php">Production Settings</a>
      <a class="btn btn-success btn-sm"
         href="?<?= http_build_query(array_merge($_GET, ['export'=>'csv'])) ?>">Export CSV</a>
    </div>
  </div>

  <div class="row g-2 mb-3">
    <div class="col-12 col-lg-8">
      <div class="btn-group btn-group-sm" role="group" aria-label="range">
        <a class="btn btn-outline-primary <?= (!$useCustom && $range==='day')?'active':'' ?>" href="?range=day">Today</a>
        <a class="btn btn-outline-primary <?= (!$useCustom && $range==='week')?'active':'' ?>" href="?range=week">7 days</a>
        <a class="btn btn-outline-primary <?= (!$useCustom && $range==='month')?'active':'' ?>" href="?range=month">30 days</a>
        <a class="btn btn-outline-primary <?= (!$useCustom && $range==='all')?'active':'' ?>" href="?range=all">All</a>
      </div>

      <form class="d-inline-flex align-items-center gap-2 ms-3" method="get">
        <input type="hidden" name="range" value="<?= htmlspecialchars($range) ?>">
        <input type="date" name="from" class="form-control form-control-sm" value="<?= htmlspecialchars($from) ?>">
        <span class="text-muted small">to</span>
        <input type="date" name="to" class="form-control form-control-sm" value="<?= htmlspecialchars($to) ?>">
        <button class="btn btn-outline-secondary btn-sm">Apply</button>
      </form>
    </div>

    <div class="col-12 col-lg-4 d-flex justify-content-lg-end align-items-center">
      <div class="text-end">
        <div class="small text-muted">Auto-refresh</div>
        <div class="fw-semibold"><?= (int)$refreshInterval ?>s</div>
      </div>
      <div class="ms-3">
        <?php if ($lastScanTs === null): ?>
          <span class="badge bg-secondary px-3 py-2">No scans</span>
        <?php else: ?>
          <?php if ($lineStopped): ?>
            <span class="badge bg-danger px-3 py-2">Line stopped (<?= (int)round($ageSeconds/60) ?>m)</span>
          <?php else: ?>
            <span class="badge bg-success px-3 py-2">Live (<?= (int)$ageSeconds ?>s)</span>
          <?php endif; ?>
        <?php endif; ?>
      </div>
    </div>
  </div>

  <div class="row g-3 mb-4">
    <div class="col-md-3">
      <div class="card text-center h-100">
        <div class="card-body">
          <div class="text-muted small">Total</div>
          <div class="display-6"><?= (int)$kpi['total'] ?></div>
        </div>
      </div>
    </div>
    <?php foreach (['keyence'=>'success','symcode'=>'warning','manual'=>'secondary'] as $src=>$cls): ?>
    <div class="col-md-3">
      <div class="card text-center h-100">
        <div class="card-body">
          <div class="text-muted small"><?= htmlspecialchars(strtoupper($src)) ?></div>
          <div class="display-6"><?= (int)$kpi[$src] ?></div>
          <div class="small">
            <span class="badge bg-<?= $cls ?>"><?= htmlspecialchars($src) ?></span>
          </div>
        </div>
      </div>
    </div>
    <?php endforeach; ?>
  </div>

  <div class="row g-3 mb-4">
    <div class="col-lg-8">
      <div class="card h-100">
        <div class="card-header d-flex justify-content-between align-items-center">
          <span>Production by source (stacked)</span>
          <span class="text-muted small"><?= $groupMode === 'hour' ? 'Hourly' : 'Daily' ?></span>
        </div>
        <div class="card-body" style="height:320px;">
          <canvas id="stackedChart"></canvas>
        </div>
      </div>
    </div>

    <div class="col-lg-4">
      <div class="card h-100">
        <div class="card-header">Scanner mode & test</div>
        <div class="card-body">

          <div class="d-flex align-items-center justify-content-between mb-2">
            <div class="fw-semibold">Active input mode</div>
            <div class="btn-group btn-group-sm" role="group">
              <button type="button" class="btn btn-outline-success" id="modeScannerBtn">Scanner</button>
              <button type="button" class="btn btn-outline-warning" id="modeManualBtn">Manual</button>
            </div>
          </div>

          <div class="small text-muted mb-2">
            Scanner = Symcode input only. Manual = PC keyboard only.
          </div>

          <div class="mb-2">
            <label class="form-label mb-1">Active Symcode reader (for identification)</label>
            <select id="readerSelect" class="form-select form-select-sm">
              <option value="">(not set)</option>
              <?php foreach ($readers as $r): ?>
                <option value="<?= htmlspecialchars($r['code']) ?>"><?= htmlspecialchars($r['code'].' — '.$r['label']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>

          <div class="mb-2">
            <label class="form-label mb-1">Symcode scanner input</label>
            <input id="symcodeInput" class="form-control form-control-sm" placeholder="Scan here (Symcode)" autocomplete="off" inputmode="none">
            <div class="mt-2 d-flex gap-2">
              <span class="badge bg-success" id="symBadgeOk" style="display:none;">SCAN OK</span>
              <span class="badge bg-danger" id="symBadgeDup" style="display:none;">DUPLICATE (ignored)</span>
              <span class="badge bg-warning text-dark" id="symBadgeBlock" style="display:none;">BLOCKED</span>
            </div>
          </div>

          <div class="mb-3">
            <label class="form-label mb-1">Manual input (PC keyboard)</label>
            <form method="post" class="d-flex gap-2" id="manualForm">
              <input name="manual_barcode" id="manualInput" class="form-control form-control-sm" placeholder="Type barcode..." autocomplete="off">
              <button class="btn btn-primary btn-sm">Add</button>
            </form>
          </div>

          <hr>

          <div class="fw-semibold mb-2">Simulator</div>
          <div class="row g-2">
            <div class="col-7">
              <input id="simCode" class="form-control form-control-sm" placeholder="code">
            </div>
            <div class="col-5">
              <select id="simSource" class="form-select form-select-sm">
                <option value="symcode">symcode</option>
                <option value="keyence">keyence</option>
                <option value="manual">manual</option>
              </select>
            </div>
          </div>
          <button class="btn btn-outline-dark btn-sm mt-2" id="simSend">Send simulated scan</button>

          <div class="small text-muted mt-2">
            Beeps: OK = high tone, Duplicate = short double, Blocked = low tone.
          </div>

        </div>
      </div>
    </div>
  </div>

  <div class="card mb-4">
    <div class="card-header d-flex justify-content-between align-items-center flex-wrap gap-2">
      <div>
        <strong>Top SKU</strong>
        <div class="small text-muted">Click a row to drill down by source.</div>
      </div>
    </div>
    <div class="card-body table-responsive">
      <table class="table table-sm table-striped align-middle mb-0">
        <thead>
          <tr>
            <th style="width:14%">SKU</th>
            <th>Description</th>
            <th class="text-end" style="width:10%">Total</th>
            <th class="text-end" style="width:10%">Keyence</th>
            <th class="text-end" style="width:10%">Symcode</th>
            <th class="text-end" style="width:10%">Manual</th>
          </tr>
        </thead>
        <tbody>
          <?php if (empty($topSku)): ?>
            <tr><td colspan="6" class="text-center text-muted">No data.</td></tr>
          <?php else: ?>
            <?php foreach ($topSku as $row): ?>
              <tr class="sku-row"
                  data-sku="<?= htmlspecialchars($row['SKU'], ENT_QUOTES) ?>"
                  data-desc="<?= htmlspecialchars($row['description'], ENT_QUOTES) ?>">
                <td><code><?= htmlspecialchars($row['SKU']) ?></code></td>
                <td><?= htmlspecialchars($row['description']) ?></td>
                <td class="text-end fw-semibold"><?= (int)$row['total'] ?></td>
                <td class="text-end"><?= (int)$row['keyence'] ?></td>
                <td class="text-end"><?= (int)$row['symcode'] ?></td>
                <td class="text-end"><?= (int)$row['manual'] ?></td>
              </tr>
            <?php endforeach; ?>
          <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>

  <div class="row g-3 mb-4">
    <?php foreach (['keyence'=>'success','symcode'=>'warning','manual'=>'secondary'] as $src=>$cls): ?>
      <div class="col-lg-4">
        <div class="card h-100">
          <div class="card-header d-flex justify-content-between align-items-center">
            <span><?= htmlspecialchars(strtoupper($src)) ?> → Top SKU</span>
            <span class="badge bg-<?= $cls ?>"><?= htmlspecialchars($src) ?></span>
          </div>
          <div class="card-body">
            <?php if (empty($sourceTopSku[$src])): ?>
              <div class="text-muted small">No data.</div>
            <?php else: ?>
              <table class="table table-sm mb-0">
                <thead><tr><th>SKU</th><th class="text-end">Boxes</th></tr></thead>
                <tbody>
                <?php foreach ($sourceTopSku[$src] as $r): ?>
                  <tr>
                    <td><code><?= htmlspecialchars($r['SKU']) ?></code></td>
                    <td class="text-end"><?= (int)$r['count'] ?></td>
                  </tr>
                <?php endforeach; ?>
                </tbody>
              </table>
            <?php endif; ?>
          </div>
        </div>
      </div>
    <?php endforeach; ?>
  </div>

  <div class="card">
    <div class="card-header d-flex justify-content-between align-items-center flex-wrap gap-2">
      <span>Live feed (latest 250)</span>
      <span class="text-muted small">Shows source + reader + SKU fields</span>
    </div>
    <div class="card-body table-responsive">
      <table class="table table-sm table-striped align-middle mb-0">
        <thead>
          <tr>
            <th style="width:16%">Time</th>
            <th style="width:10%">Source</th>
            <th style="width:18%">Reader</th>
            <th style="width:18%">Barcode</th>
            <th style="width:10%">SKU</th>
            <th>Grower</th>
          </tr>
        </thead>
        <tbody>
          <?php if (empty($live)): ?>
            <tr><td colspan="6" class="text-center text-muted">No scans.</td></tr>
          <?php else: ?>
            <?php foreach ($live as $r): ?>
              <?php
                $src = (string)($r['source'] ?? '');
                $badge = ($src === 'keyence') ? 'success' : (($src === 'symcode') ? 'warning' : 'secondary');
              ?>
              <tr>
                <td><?= htmlspecialchars((string)$r['scanned_at']) ?></td>
                <td><span class="badge bg-<?= $badge ?>"><?= htmlspecialchars($src) ?></span></td>
                <td><?php else: ?>
                    <span class="text-muted small">(n/a)</span>
                  <?php endif; ?>
                </td>
                <td><code><?= htmlspecialchars((string)$r['code']) ?></code></td>
                <td><?= htmlspecialchars((string)($r['SKU'] ?? '')) ?></td>
                <td><?= htmlspecialchars((string)($r['grower'] ?? '')) ?></td>
              </tr>
            <?php endforeach; ?>
          <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>

</div>

<div class="modal fade" id="skuModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="skuModalTitle">SKU</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <div class="row g-3">
          <div class="col-md-4">
            <div class="border rounded p-3 h-100">
              <div class="text-muted small mb-1">By source</div>
              <div id="skuModalCounts"></div>
            </div>
          </div>
          <div class="col-md-8">
            <div class="border rounded p-3 h-100">
              <div class="text-muted small mb-2">Recent scans for this SKU (from live list)</div>
              <div class="table-responsive">
                <table class="table table-sm mb-0" id="skuModalTable">
                  <thead><tr><th>Time</th><th>Source</th><th>Reader</th><th>Barcode</th></tr></thead>
                  <tbody></tbody>
                </table>
              </div>
              <div class="small text-muted mt-2">Dedup is enforced by UNIQUE(code). Duplicate attempts are ignored and logged.</div>
            </div>
          </div>
        </div>
      </div>
      <div class="modal-footer">
        <button class="btn btn-secondary btn-sm" data-bs-dismiss="modal">Close</button>
      </div>
    </div>
  </div>
</div>

<?php include '../includes/footer.php'; ?>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
const REFRESH_MS = <?= (int)$refreshInterval * 1000 ?>;

const chartLabels   = <?= json_encode($stackLabels, JSON_UNESCAPED_UNICODE) ?>;
const dataKeyence   = <?= json_encode($stackKeyence, JSON_NUMERIC_CHECK) ?>;
const dataSymcode   = <?= json_encode($stackSymcode, JSON_NUMERIC_CHECK) ?>;
const dataManual    = <?= json_encode($stackManual, JSON_NUMERIC_CHECK) ?>;

(function(){
  const canvas = document.getElementById('stackedChart');
  if (!canvas || chartLabels.length === 0) return;

  new Chart(canvas.getContext('2d'), {
    type: 'bar',
    data: {
      labels: chartLabels,
      datasets: [
        { label: 'keyence', data: dataKeyence, stack: 'total' },
        { label: 'symcode', data: dataSymcode, stack: 'total' },
        { label: 'manual',  data: dataManual,  stack: 'total' },
      ]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      scales: {
        x: { stacked: true, ticks: { autoSkip: true, maxTicksLimit: 16 } },
        y: { stacked: true, beginAtZero: true, precision: 0 }
      }
    }
  });
})();

let audioCtx = null;
function beep(freq, ms) {
  try {
    if (!audioCtx) audioCtx = new (window.AudioContext || window.webkitAudioContext)();
    const o = audioCtx.createOscillator();
    const g = audioCtx.createGain();
    o.type = 'sine';
    o.frequency.value = freq;
    g.gain.value = 0.08;
    o.connect(g); g.connect(audioCtx.destination);
    o.start();
    setTimeout(()=>{ o.stop(); }, ms);
  } catch(e) {}
}
function beepOK(){ beep(880, 110); }
function beepDup(){ beep(220, 80); setTimeout(()=>beep(220, 80), 120); }
function beepBlock(){ beep(140, 180); }

const modeScannerBtn = document.getElementById('modeScannerBtn');
const modeManualBtn  = document.getElementById('modeManualBtn');
const symInput       = document.getElementById('symcodeInput');
const manualInput    = document.getElementById('manualInput');

function setMode(mode) {
  localStorage.setItem('ps_mode', mode);
  if (mode === 'manual') {
    modeManualBtn.classList.add('active');
    modeScannerBtn.classList.remove('active');
    symInput.disabled = true;
    manualInput.disabled = false;
    manualInput.focus();
  } else {
    modeScannerBtn.classList.add('active');
    modeManualBtn.classList.remove('active');
    symInput.disabled = false;
    manualInput.disabled = true;
    symInput.focus();
  }
}
setMode(localStorage.getItem('ps_mode') || 'scanner');
modeScannerBtn.addEventListener('click', ()=>setMode('scanner'));
modeManualBtn.addEventListener('click', ()=>setMode('manual'));

const readerSelect = document.getElementById('readerSelect');
const savedReader = localStorage.getItem('ps_reader') || '';
if (savedReader) readerSelect.value = savedReader;
readerSelect.addEventListener('change', ()=> {
  localStorage.setItem('ps_reader', readerSelect.value || '');
});

const bOk = document.getElementById('symBadgeOk');
const bDup = document.getElementById('symBadgeDup');
const bBlock = document.getElementById('symBadgeBlock');
function showBadge(which) {
  [bOk,bDup,bBlock].forEach(b=>b.style.display='none');
  if (which==='ok') bOk.style.display='inline-block';
  if (which==='dup') bDup.style.display='inline-block';
  if (which==='block') bBlock.style.display='inline-block';
  setTimeout(()=>{ [bOk,bDup,bBlock].forEach(b=>b.style.display='none'); }, 1500);
}

async function ingest(code, source) {
  const mode = localStorage.getItem('ps_mode') || 'scanner';
  if (source === 'symcode' && mode !== 'scanner') {
    showBadge('block'); beepBlock(); return {status:'blocked'};
  }
  if (source === 'manual' && mode !== 'manual') {
    showBadge('block'); beepBlock(); return {status:'blocked'};
  }

  const payload = {
    code: code,
    source: source,
    scan_function: 'case_count',{
    method: 'POST',
    headers: {'Content-Type':'application/json'},
    body: JSON.stringify(payload)
  });
  let j = null;
  try { j = await r.json(); } catch(e) { j = {status:'error'}; }
  return j;
}

symInput.addEventListener('keydown', async (e) => {
  if (e.key !== 'Enter') return;
  e.preventDefault();
  const code = (symInput.value || '').trim();
  if (!code) return;

  symInput.value = '';
  const res = await ingest(code, 'symcode');
  if (res.status === 'inserted') { showBadge('ok'); beepOK(); }
  else if (res.status === 'duplicate') { showBadge('dup'); beepDup(); }
  else { showBadge('block'); beepBlock(); }

  setTimeout(()=>location.reload(), 600);
});

document.getElementById('simSend').addEventListener('click', async () => {
  const code = (document.getElementById('simCode').value || '').trim();
  const src  = document.getElementById('simSource').value;
  if (!code) return;
  const res = await ingest(code, src);
  if (res.status === 'inserted') beepOK();
  else if (res.status === 'duplicate') beepDup();
  else beepBlock();
  setTimeout(()=>location.reload(), 600);
});

const skuSource = <?= json_encode($skuSource, JSON_UNESCAPED_UNICODE | JSON_NUMERIC_CHECK) ?>;
const liveRows  = <?= json_encode($live, JSON_UNESCAPED_UNICODE | JSON_NUMERIC_CHECK) ?>;

document.querySelectorAll('.sku-row').forEach(tr => {
  tr.addEventListener('click', () => {
    const sku = tr.getAttribute('data-sku') || '(n/a)';
    const desc = tr.getAttribute('data-desc') || '';
    const counts = skuSource[sku] || {keyence:0,symcode:0,manual:0,total:0};

    document.getElementById('skuModalTitle').textContent = `SKU ${sku} — ${desc}`;
    document.getElementById('skuModalCounts').innerHTML =
      `<div class="mb-1"><span class="badge bg-success">keyence</span> <span class="float-end">${counts.keyence||0}</span></div>` +
      `<div class="mb-1"><span class="badge bg-warning text-dark">symcode</span> <span class="float-end">${counts.symcode||0}</span></div>` +
      `<div class="mb-1"><span class="badge bg-secondary">manual</span> <span class="float-end">${counts.manual||0}</span></div>` +
      `<hr class="my-2"><div class="fw-semibold">Total <span class="float-end">${counts.total||0}</span></div>`;

    const tbody = document.querySelector('#skuModalTable tbody');
    tbody.innerHTML = '';
    const rows = liveRows.filter(r => (r.SKU || '(n/a)') === sku).slice(0, 25);
    rows.forEach(r => {
      const badge = (r.source==='keyence') ? 'success' : (r.source==='symcode' ? 'warning' : 'secondary');
            const rowEl = document.createElement('tr');
      rowEl.innerHTML =
        `<td>${r.scanned_at}</td>` +
        `<td><span class="badge bg-${badge}">${r.source}</span></td>` +
        `<td><small>${reader}</small></td>` +
        `<td><code>${r.code}</code></td>`;
      tbody.appendChild(rowEl);
    });

    const modalEl = document.getElementById('skuModal');
    const modal = new bootstrap.Modal(modalEl);
    modal.show();
  });
});

setTimeout(()=>location.reload(), REFRESH_MS);
</script>
