<?php
require_once __DIR__ . '/../includes/db.php';

header('Content-Type: application/json');

$raw = file_get_contents('php://input');
$payload = json_decode($raw, true);
if (!is_array($payload)) $payload = $_POST;

$code = trim((string)($payload['code'] ?? ''));
$expected = trim((string)($payload['scan_function'] ?? $payload['expected_function'] ?? 'case_count'));
$source = trim((string)($payload['source'] ?? 'symcode'));

$dryRun = !empty($payload['dry_run']);

if ($code === '') {
    echo json_encode(['status'=>'blocked','message'=>'Empty code']); exit;
}
if (!in_array($expected, ['case_count','dump_bins','palletize'], true)) $expected = 'case_count';
if (!in_array($source, ['keyence','symcode','manual'], true)) $source = 'symcode';

$profilesFile = __DIR__ . '/../config/scanner_profiles.json';
$cfg = [];
if (is_file($profilesFile)) {
    $j = json_decode(@file_get_contents($profilesFile), true);
    if (is_array($j)) $cfg = $j;
}
$enforce = !empty(($cfg['mode']['enforce_block'] ?? false));
$profiles = $cfg['profiles'] ?? [];

$active = trim((string)($payload['active_scanner_id'] ?? ''));
$force = trim((string)($payload['force_scanner_id'] ?? ''));

function sp_match_profile($profiles, $code, $active, $force) {
    if ($force !== '') {
        foreach ($profiles as $p) if (($p['id'] ?? '') === $force) return $p;
    }
    foreach ($profiles as $p) {
        if (empty($p['enabled'])) continue;
        $prefix = (string)($p['identifier_prefix'] ?? '');
        $suffix = (string)($p['identifier_suffix'] ?? '');
        if ($prefix === '' && $suffix === '') continue;
        $ok = true;
        if ($prefix !== '' && stripos($code, $prefix) !== 0) $ok = false;
        if ($suffix !== '' && $ok) {
            $len = strlen($suffix);
            if ($len > 0 && substr($code, -$len) !== $suffix) $ok = false;
        }
        if ($ok) return $p;
    }
    if ($active !== '') {
        foreach ($profiles as $p) if (($p['id'] ?? '') === $active) return $p;
    }
    return null;
}

$matched = sp_match_profile($profiles, $code, $active, $force);
$allowed = $matched['allowed_functions'] ?? [];
if (!is_array($allowed)) $allowed = [];
$beep = $matched['beep'] ?? 'mid';
$matchedLabel = $matched['label'] ?? null;
$matchedType = $matched['type'] ?? null;

if ($matchedType === 'keyence') $source = 'keyence';
elseif ($matchedType === 'manual') $source = 'manual';
elseif ($matchedType === 'symcode') $source = 'symcode';

if ($matched && $enforce && !in_array($expected, $allowed, true)) {
    echo json_encode(['status'=>'blocked','message'=>'Blocked: scanner not allowed for this function','beep'=>'low','matched_label'=>$matchedLabel]); exit;
}

$logDir = __DIR__ . '/../logs';
if (!is_dir($logDir)) @mkdir($logDir, 0775, true);
@file_put_contents($logDir . '/scanner_audit.log', date('c') . " " . ($dryRun?'DRYRUN':'IN') . " code={$code} source={$source} func={$expected} matched=" . ($matched['id'] ?? 'none') . "\n", FILE_APPEND);

if ($dryRun) {
    echo json_encode([
        'status'=>'ok',
        'message'=>'Dry run',
        'beep'=>$beep,
        'matched_id'=>$matched['id'] ?? null,
        'matched_label'=>$matchedLabel,
        'matched_type'=>$matchedType,
        'allowed_functions'=>$allowed,
        'expected_function'=>$expected
    ]);
    exit;
}

try {
    $stmt = $pdo->prepare("INSERT INTO barcode_scans (code, source, scan_function) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE id=id");
    $stmt->execute([$code, $source, $expected]);
    $ins = ($stmt->rowCount() > 0);
    echo json_encode([
        'status'=>$ins?'ok':'dup',
        'message'=>$ins?'Inserted':'Duplicate ignored',
        'beep'=>$beep,
        'source'=>$source,
        'scan_function'=>$expected
    ]);
} catch (Throwable $e) {
    echo json_encode(['status'=>'blocked','message'=>'DB error: '.$e->getMessage(),'beep'=>'low']);
}
