<?php
require_once __DIR__ . '/../config/user_functions.php';

if (session_status() === PHP_SESSION_NONE) { session_start(); }
require_once __DIR__ . '/../config/users_local.php';

// centralized email helpers (Gmail + JSON config)
require_once __DIR__ . '/../config/emailer.php';

if (!user_has_permission('settings')) {
    http_response_code(403);
    include '../includes/header.php';
    include '../includes/sidebar.php';
    echo "<div class='container-fluid py-4'><h3 class='text-danger'>Access denied</h3></div>";
    include '../includes/footer.php';
    exit;
}

// ------------------------------------------------------
// LOAD EMAIL / ALERT CONFIG (through helpers in emailer.php)
// ------------------------------------------------------
$emailCfg        = sp_email_get_settings();
$emailRecipients = ['to' => sp_email_get_recipients()];
$alertsCfg       = sp_email_get_alerts();

// Flash messages
$emailFlash  = '';
$recipFlash  = '';
$alertsFlash = '';

// ------------------------------------------------------
// HANDLE POST (SMTP / RECIPIENTS / ALERTS)
// ------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    // --- EMAIL / SMTP SETTINGS + TEST ---
    if (isset($_POST['email_action'])) {
        $action = $_POST['email_action'];

        if ($action === 'save_smtp' || $action === 'test_smtp') {
            $emailCfg['smtp_host'] = trim($_POST['smtp_host'] ?? '');
            $emailCfg['smtp_port'] = (int)($_POST['smtp_port'] ?? 587);
            if ($emailCfg['smtp_port'] <= 0) {
                $emailCfg['smtp_port'] = 587;
            }

            $emailCfg['smtp_user'] = trim($_POST['smtp_user'] ?? '');
            $smtpPass = $_POST['smtp_pass'] ?? '';
            if ($smtpPass !== '') {
                $emailCfg['smtp_pass'] = $smtpPass;
            }

            $secure = strtolower(trim($_POST['smtp_secure'] ?? 'tls'));
            if (!in_array($secure, ['tls', 'ssl', 'none'], true)) {
                $secure = 'tls';
            }
            $emailCfg['smtp_secure'] = $secure;

            $emailCfg['from_email'] = trim($_POST['from_email'] ?? '');
            if ($emailCfg['from_email'] === '') {
                $emailCfg['from_email'] = 'itsmproduce@gmail.com';
            }

            $emailCfg['from_name'] = trim($_POST['from_name'] ?? '');
            if ($emailCfg['from_name'] === '') {
                $emailCfg['from_name'] = 'SM Produce Reports';
            }

            sp_email_save_settings($emailCfg);
            $emailFlash = 'SMTP settings saved.';

            // TEST email via Gmail / PHPMailer
            if ($action === 'test_smtp') {
                $testTo = trim($_POST['test_email_to'] ?? $emailCfg['from_email']);
                if ($testTo !== '') {
                    $subject = 'Test email from SM Produce';
                    $body    = "This is a test email from SM Produce settings.<br><br>"
                             . "If you received this, Gmail SMTP is working.";
                    $ok = sp_smtp_send($subject, $body, [$testTo]);
                    if ($ok) {
                        $emailFlash = "Test email sent via Gmail SMTP to {$testTo}.";
                    } else {
                        $emailFlash = "Failed to send test email via Gmail SMTP. Check credentials and Gmail app password.";
                    }
                } else {
                    $emailFlash = 'Test email address is empty.';
                }
            }
        }

        // --- ADD RECIPIENT ---
        if ($action === 'add_recipient') {
            $new = trim($_POST['new_recipient'] ?? '');
            if ($new !== '' && filter_var($new, FILTER_VALIDATE_EMAIL)) {
                if (!in_array($new, $emailRecipients['to'], true)) {
                    $emailRecipients['to'][] = $new;
                    sp_email_set_recipients($emailRecipients['to']);
                    $recipFlash = "Recipient {$new} added.";
                } else {
                    $recipFlash = "Recipient {$new} already exists.";
                }
            } else {
                $recipFlash = 'Invalid email address.';
            }
        }

        // --- REMOVE RECIPIENT ---
        if ($action === 'remove_recipient') {
            $rem = trim($_POST['recipient_email'] ?? '');
            if ($rem !== '') {
                $emailRecipients['to'] = array_values(array_filter(
                    $emailRecipients['to'],
                    static fn($e) => $e !== $rem
                ));
                sp_email_set_recipients($emailRecipients['to']);
                $recipFlash = "Recipient {$rem} removed.";
            }
        }
    }

    // --- ALERT TIMEOUT SETTINGS ---
    if (isset($_POST['alerts_action']) && $_POST['alerts_action'] === 'save_alerts') {
        $alertsCfg['enabled'] = !empty($_POST['alerts_enabled']);
        $tm = (int)($_POST['timeout_minutes'] ?? 60);
        if ($tm <= 0) {
            $tm = 60;
        }
        $alertsCfg['timeout_minutes'] = $tm;
        sp_email_save_alerts($alertsCfg);
        $alertsFlash = 'Timeout alert settings saved.';
    }
}

include '../includes/header.php';
include '../includes/sidebar.php';
?>
<div class="container-fluid py-4">
  <h3 class="mb-4">Settings</h3>
  <div class="row g-4">

    <!-- PRINTER SETTINGS -->
    <div class="col-md-4">
      <div class="card h-100">
        <div class="card-body">
          <h5 class="card-title">Printers</h5>
          <p class="card-text">Configure default printers and printing behavior.</p>
          <a href="/smproduce/pages/settings_printers.php" class="btn btn-outline-primary">Printer Settings</a>
        </div>
      </div>
    </div>

    <!-- FULL BIN LABEL SETTINGS -->
    <div class="col-md-4">
      <div class="card h-100">
        <div class="card-body">
          <h5 class="card-title">Full Bin Labels</h5>
          <p class="card-text">Choose which fields appear on full bin labels.</p>
          <a href="/smproduce/pages/settings_labels.php" class="btn btn-outline-primary">Label Options</a>
        </div>
      </div>
    </div>

    <!-- REPORT SETTINGS -->
    <div class="col-md-4">
      <div class="card h-100">
        <div class="card-body">
          <h5 class="card-title">Reports</h5>
          <p class="card-text">Select default printer and which columns to include in reports.</p>
          <a href="/smproduce/pages/settings_reports.php" class="btn btn-outline-primary">Report Options</a>
        </div>
      </div>
    </div>

    <!-- ORDERS PRINT SETTINGS -->
    <div class="col-md-4">
      <div class="card h-100">
        <div class="card-body">
          <h5 class="card-title">Orders Printing</h5>
          <p class="card-text">Choose printer and auto-print options for orders.</p>
          <a href="/smproduce/pages/settings_orders.php" class="btn btn-outline-primary">Orders Printer</a>
        </div>
      </div>
    </div>

    <!-- PRODUCTION SETTINGS -->
    <div class="col-md-4">
      <div class="card h-100 border-info">
        <div class="card-body">
          <h5 class="card-title">Production Settings</h5>
          <p class="card-text">Configure production summary fields, charts and refresh options.</p>
          <a href="/smproduce/pages/settings_production.php" class="btn btn-outline-info">
            Production Settings
          </a>
        </div>
      </div>
    </div>

  </div>


<?php
if (!function_exists('sp_json_read')) {
  function sp_json_read($path, $default = []) {
      if (!is_file($path)) return $default;
      $raw = @file_get_contents($path);
      if ($raw === false) return $default;
      $j = json_decode($raw, true);
      return is_array($j) ? $j : $default;
  }
}
if (!function_exists('sp_json_write')) {
  function sp_json_write($path, $data) {
      $dir = dirname($path);
      if (!is_dir($dir)) @mkdir($dir, 0775, true);
      @file_put_contents($path, json_encode($data, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE));
  }
}
if (!function_exists('sp_h')) {
  function sp_h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
}

$profilesFile = __DIR__ . '/../config/scanner_profiles.json';
$profilesCfg = sp_json_read($profilesFile, [
  'profiles' => [
    ['id'=>'symcode_prod','type'=>'symcode','label'=>'Symcode - Production (Cases)','enabled'=>true,'allowed_functions'=>['case_count'],'identifier_prefix'=>'','identifier_suffix'=>'','badge'=>'warning','beep'=>'mid'],
    ['id'=>'symcode_dump','type'=>'symcode','label'=>'Symcode - Dumping (Bins)','enabled'=>true,'allowed_functions'=>['dump_bins'],'identifier_prefix'=>'','identifier_suffix'=>'','badge'=>'warning','beep'=>'high'],
    ['id'=>'symcode_pallet','type'=>'symcode','label'=>'Symcode - Palletize','enabled'=>true,'allowed_functions'=>['palletize'],'identifier_prefix'=>'','identifier_suffix'=>'','badge'=>'warning','beep'=>'low'],
    ['id'=>'keyence_prod','type'=>'keyence','label'=>'Keyence - Production','enabled'=>true,'allowed_functions'=>['case_count'],'identifier_prefix'=>'','identifier_suffix'=>'','badge'=>'primary','beep'=>'high'],
    ['id'=>'manual_pc','type'=>'manual','label'=>'Manual (PC Keyboard)','enabled'=>true,'allowed_functions'=>['case_count'],'identifier_prefix'=>'','identifier_suffix'=>'','badge'=>'secondary','beep'=>'mid']
  ],
  'mode' => ['enforce_block'=>true]
]);

$scannersFlash = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_scanner_profiles'])) {
  $profilesCfg['mode']['enforce_block'] = !empty($_POST['enforce_block']);
  $incoming = $_POST['p'] ?? [];
  $seen = [];
  $newProfiles = [];
  if (is_array($incoming)) {
    foreach ($incoming as $p) {
      $id = trim((string)($p['id'] ?? ''));
      if ($id === '' || isset($seen[$id])) continue;
      $seen[$id] = true;
      $type = (string)($p['type'] ?? 'symcode');
      if (!in_array($type, ['symcode','keyence','manual'], true)) $type = 'symcode';
      $label = trim((string)($p['label'] ?? $id));
      if ($label === '') $label = $id;
      $enabled = !empty($p['enabled']);
      $badge = (string)($p['badge'] ?? 'secondary');
      if (!in_array($badge, ['primary','secondary','success','warning','danger','info','dark'], true)) $badge = 'secondary';
      $beep = (string)($p['beep'] ?? 'mid');
      if (!in_array($beep, ['high','mid','low','none'], true)) $beep = 'mid';
      $prefix = trim((string)($p['identifier_prefix'] ?? ''));
      $suffix = trim((string)($p['identifier_suffix'] ?? ''));
      $allowed = $p['allowed_functions'] ?? [];
      if (!is_array($allowed)) $allowed = [];
      $allowed = array_values(array_unique(array_filter(array_map('trim', $allowed), function($x){
        return in_array($x, ['case_count','dump_bins','palletize'], true);
      })));
      if (empty($allowed)) $allowed = ['case_count'];
      $newProfiles[] = ['id'=>$id,'type'=>$type,'label'=>$label,'enabled'=>$enabled,'allowed_functions'=>$allowed,'identifier_prefix'=>$prefix,'identifier_suffix'=>$suffix,'badge'=>$badge,'beep'=>$beep];
    }
  }
  if (!empty($newProfiles)) $profilesCfg['profiles'] = $newProfiles;
  sp_json_write($profilesFile, $profilesCfg);
  $scannersFlash = 'Symcode profiles saved.';
}
?>

<div class="card mt-4 border-dark">
  <div class="card-header bg-dark text-white"><strong>Symcode Profiles & Routing</strong></div>
  <div class="card-body">
    <?php if ($scannersFlash): ?>
      <div class="alert alert-success py-2"><?= sp_h($scannersFlash) ?></div>
    <?php endif; ?>

    <form method="post">
      <input type="hidden" name="save_scanner_profiles" value="1">
      <div class="form-check mb-3">
        <input class="form-check-input" type="checkbox" id="enforce_block" name="enforce_block" value="1" <?= !empty(($profilesCfg['mode']['enforce_block'] ?? false)) ? 'checked' : '' ?>>
        <label class="form-check-label" for="enforce_block">Block wrong scanner/function</label>
      </div>

      <div class="table-responsive">
        <table class="table table-sm align-middle">
          <thead>
            <tr>
              <th style="width:12%">ID</th>
              <th style="width:10%">Type</th>
              <th style="width:20%">Label</th>
              <th style="width:18%">Allowed functions</th>
              <th style="width:12%">Prefix</th>
              <th style="width:12%">Suffix</th>
              <th style="width:8%">Badge</th>
              <th style="width:8%">Beep</th>
              <th style="width:6%" class="text-center">On</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach (($profilesCfg['profiles'] ?? []) as $i => $p): ?>
              <?php $af = $p['allowed_functions'] ?? []; if (!is_array($af)) $af = []; ?>
              <tr>
                <td><input type="text" name="p[<?= (int)$i ?>][id]" value="<?= sp_h($p['id'] ?? '') ?>" class="form-control form-control-sm"></td>
                <td>
                  <select name="p[<?= (int)$i ?>][type]" class="form-select form-select-sm">
                    <?php $t = $p['type'] ?? 'symcode'; ?>
                    <option value="symcode" <?= $t==='symcode'?'selected':'' ?>>symcode</option>
                    <option value="keyence" <?= $t==='keyence'?'selected':'' ?>>keyence</option>
                    <option value="manual"  <?= $t==='manual'?'selected':'' ?>>manual</option>
                  </select>
                </td>
                <td><input type="text" name="p[<?= (int)$i ?>][label]" value="<?= sp_h($p['label'] ?? '') ?>" class="form-control form-control-sm"></td>
                <td>
                  <select name="p[<?= (int)$i ?>][allowed_functions][]" multiple class="form-select form-select-sm" size="3">
                    <option value="case_count" <?= in_array('case_count',$af,true)?'selected':'' ?>>case_count</option>
                    <option value="dump_bins" <?= in_array('dump_bins',$af,true)?'selected':'' ?>>dump_bins</option>
                    <option value="palletize" <?= in_array('palletize',$af,true)?'selected':'' ?>>palletize</option>
                  </select>
                </td>
                <td><input type="text" name="p[<?= (int)$i ?>][identifier_prefix]" value="<?= sp_h($p['identifier_prefix'] ?? '') ?>" class="form-control form-control-sm"></td>
                <td><input type="text" name="p[<?= (int)$i ?>][identifier_suffix]" value="<?= sp_h($p['identifier_suffix'] ?? '') ?>" class="form-control form-control-sm"></td>
                <td>
                  <?php $b = $p['badge'] ?? 'secondary'; ?>
                  <select name="p[<?= (int)$i ?>][badge]" class="form-select form-select-sm">
                    <?php foreach (['primary','secondary','success','warning','danger','info','dark'] as $opt): ?>
                      <option value="<?= $opt ?>" <?= $b===$opt?'selected':'' ?>><?= $opt ?></option>
                    <?php endforeach; ?>
                  </select>
                </td>
                <td>
                  <?php $bp = $p['beep'] ?? 'mid'; ?>
                  <select name="p[<?= (int)$i ?>][beep]" class="form-select form-select-sm">
                    <option value="high" <?= $bp==='high'?'selected':'' ?>>high</option>
                    <option value="mid"  <?= $bp==='mid'?'selected':'' ?>>mid</option>
                    <option value="low"  <?= $bp==='low'?'selected':'' ?>>low</option>
                    <option value="none" <?= $bp==='none'?'selected':'' ?>>none</option>
                  </select>
                </td>
                <td class="text-center"><input type="checkbox" name="p[<?= (int)$i ?>][enabled]" value="1" <?= !empty($p['enabled'])?'checked':'' ?>></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>

      <button class="btn btn-primary btn-sm">Save profiles</button>
    </form>

    <hr>

    <div class="row g-3">
      <div class="col-lg-6">
        <div class="border rounded p-3">
          <div class="fw-semibold mb-2">Test identificazione (live)</div>
          <div class="row g-2 align-items-end">
            <div class="col-md-6">
              <label class="form-label mb-1">Expected function</label>
              <select id="testExpected" class="form-select form-select-sm">
                <option value="case_count">case_count</option>
                <option value="dump_bins">dump_bins</option>
                <option value="palletize">palletize</option>
              </select>
            </div>
            <div class="col-md-6">
              <label class="form-label mb-1">Active Symcode (if no prefix/suffix)</label>
              <select id="testActive" class="form-select form-select-sm">
                <option value="">(auto / unknown)</option>
                <?php foreach (($profilesCfg['profiles'] ?? []) as $p): ?>
                  <?php if (($p['type'] ?? '') === 'symcode'): ?>
                    <option value="<?= sp_h($p['id']) ?>"><?= sp_h($p['label'] ?? $p['id']) ?></option>
                  <?php endif; ?>
                <?php endforeach; ?>
              </select>
            </div>
          </div>

          <div class="mt-2">
            <label class="form-label mb-1">Scan here</label>
            <input id="testScan" class="form-control" placeholder="Scan with Symcode or type...">
            <div class="small text-muted mt-1">Press Enter after scan.</div>
          </div>

          <div class="mt-3" id="testResult"></div>
        </div>
      </div>

      <div class="col-lg-6">
        <div class="border rounded p-3">
          <div class="fw-semibold mb-2">Simulator</div>
          <div class="row g-2">
            <div class="col-md-6">
              <select id="simSource" class="form-select form-select-sm">
                <option value="symcode">symcode</option>
                <option value="manual">manual</option>
                <option value="keyence">keyence</option>
              </select>
            </div>
            <div class="col-md-6">
              <select id="simFunction" class="form-select form-select-sm">
                <option value="case_count">case_count</option>
                <option value="dump_bins">dump_bins</option>
                <option value="palletize">palletize</option>
              </select>
            </div>
          </div>
          <div class="d-flex gap-2 mt-2">
            <input id="simCode" class="form-control form-control-sm" placeholder="Code (optional)">
            <button class="btn btn-outline-primary btn-sm" type="button" id="simBtn">Simulate (insert)</button>
          </div>
          <div class="mt-2" id="simResult"></div>
        </div>
      </div>
    </div>

    <div class="small text-muted mt-3">
      Audit log: <code>/logs/scanner_audit.log</code>. Ingestion endpoint: <code>/pages/scan_ingest.php</code>.
    </div>
  </div>
</div>

<script>
function spBeep(kind){
  if(kind==='none') return;
  const ctx = new (window.AudioContext || window.webkitAudioContext)();
  const o = ctx.createOscillator();
  const g = ctx.createGain();
  let f = 750;
  if(kind==='high') f=1100;
  if(kind==='low') f=450;
  o.type='sine'; o.frequency.value=f;
  o.connect(g); g.connect(ctx.destination);
  g.gain.value=0.06;
  o.start(); setTimeout(()=>{o.stop(); ctx.close();}, 120);
}
async function spPost(body){
  const r = await fetch('/smproduce/pages/scan_ingest.php', {method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify(body)});
  return await r.json();
}
function spBadge(status){
  let cls='secondary';
  if(status==='ok') cls='success';
  if(status==='dup') cls='warning';
  if(status==='blocked') cls='danger';
  return cls;
}

document.getElementById('testScan')?.addEventListener('keydown', async (e)=>{
  if(e.key!=='Enter') return;
  e.preventDefault();
  const code = (document.getElementById('testScan')?.value || '').trim();
  if(!code) return;
  const expected_function = document.getElementById('testExpected')?.value || 'case_count';
  const active_scanner_id = document.getElementById('testActive')?.value || '';
  const res = await spPost({code, expected_function, active_scanner_id, dry_run: 1});
  const el = document.getElementById('testResult');
  if(el){
    el.innerHTML = '<div class="border rounded p-2">' +
      '<span class="badge bg-'+spBadge(res.status)+'">'+(res.status||'').toUpperCase()+'</span>' +
      '<div class="mt-2"><strong>Matched:</strong> '+(res.matched_label||'(none)')+'</div>' +
      '<div><strong>Type:</strong> '+(res.matched_type||'(none)')+'</div>' +
      '<div><strong>Expected:</strong> '+(res.expected_function||'')+'</div>' +
      '<div><strong>Allowed:</strong> '+((res.allowed_functions||[]).join(', '))+'</div>' +
      '<div><strong>Message:</strong> '+(res.message||'')+'</div>' +
    '</div>';
  }
  spBeep(res.beep || 'mid');
  document.getElementById('testScan').value='';
});

document.getElementById('simBtn')?.addEventListener('click', async ()=>{
  let code = (document.getElementById('simCode')?.value || '').trim();
  if(!code) code = 'SIM' + Math.floor(Math.random()*1000000);
  const source = document.getElementById('simSource')?.value || 'symcode';
  const scan_function = document.getElementById('simFunction')?.value || 'case_count';
  const res = await spPost({code, source, scan_function});
  const el = document.getElementById('simResult');
  if(el){
    el.innerHTML = '<div><span class="badge bg-'+spBadge(res.status)+'">'+(res.status||'').toUpperCase()+'</span> '+(res.message||'')+'</div>';
  }
  spBeep(res.beep || 'mid');
});
</script>


  <!-- FLASH MESSAGES EMAIL / ALERTS -->
  <div class="mt-4">
    <?php if ($emailFlash): ?>
      <div class="alert alert-info py-2 mb-2"><?= htmlspecialchars($emailFlash) ?></div>
    <?php endif; ?>
    <?php if ($recipFlash): ?>
      <div class="alert alert-info py-2 mb-2"><?= htmlspecialchars($recipFlash) ?></div>
    <?php endif; ?>
    <?php if ($alertsFlash): ?>
      <div class="alert alert-info py-2 mb-2"><?= htmlspecialchars($alertsFlash) ?></div>
    <?php endif; ?>
  </div>

  <!-- EMAIL & TIMEOUT ALERTS SECTION -->
  <div class="row mt-2 g-4">
    <div class="col-lg-8">
      <div class="card shadow-sm">
        <div class="card-header bg-light">
          <h5 class="mb-0">Email & Timeout Alerts</h5>
          <small class="text-muted">Gmail SMTP settings, recipients and Keyence inactivity timeout.</small>
        </div>
        <div class="card-body">

          <!-- SMTP SETTINGS -->
          <h6 class="mb-3">SMTP Settings (Gmail)</h6>
          <form method="post" class="mb-4">
            <input type="hidden" name="email_action" value="save_smtp">

            <div class="row g-3">
              <div class="col-md-6">
                <label class="form-label">SMTP Host</label>
                <input type="text" name="smtp_host" class="form-control form-control-sm"
                       value="<?= htmlspecialchars($emailCfg['smtp_host'] ?? '') ?>">
              </div>
              <div class="col-md-3">
                <label class="form-label">SMTP Port</label>
                <input type="number" name="smtp_port" class="form-control form-control-sm"
                       value="<?= htmlspecialchars((string)($emailCfg['smtp_port'] ?? 587)) ?>">
              </div>
              <div class="col-md-3">
                <label class="form-label">Encryption</label>
                <select name="smtp_secure" class="form-select form-select-sm">
                  <?php $sec = strtolower($emailCfg['smtp_secure'] ?? 'tls'); ?>
                  <option value="tls"  <?= $sec === 'tls'  ? 'selected' : '' ?>>TLS</option>
                  <option value="ssl"  <?= $sec === 'ssl'  ? 'selected' : '' ?>>SSL</option>
                  <option value="none" <?= $sec === 'none' ? 'selected' : '' ?>>None</option>
                </select>
              </div>
            </div>

            <div class="row g-3 mt-2">
              <div class="col-md-6">
                <label class="form-label">SMTP User (Gmail address)</label>
                <input type="text" name="smtp_user" class="form-control form-control-sm"
                       value="<?= htmlspecialchars($emailCfg['smtp_user'] ?? '') ?>">
              </div>
              <div class="col-md-6">
                <label class="form-label">SMTP Password (App Password)</label>
                <input type="password" name="smtp_pass" class="form-control form-control-sm"
                       placeholder="●●●●●●●● (leave blank to keep current)">
              </div>
            </div>

            <div class="row g-3 mt-2">
              <div class="col-md-6">
                <label class="form-label">From email</label>
                <input type="email" name="from_email" class="form-control form-control-sm"
                       value="<?= htmlspecialchars($emailCfg['from_email'] ?? '') ?>">
              </div>
              <div class="col-md-6">
                <label class="form-label">From name</label>
                <input type="text" name="from_name" class="form-control form-control-sm"
                       value="<?= htmlspecialchars($emailCfg['from_name'] ?? '') ?>">
              </div>
            </div>

            <div class="row g-3 mt-3 align-items-end">
              <div class="col-md-6">
                <label class="form-label">Test email to</label>
                <input type="email" name="test_email_to" class="form-control form-control-sm"
                       placeholder="you@example.com">
              </div>
              <div class="col-md-6 text-end">
                <button type="submit" class="btn btn-primary btn-sm me-2">
                  💾 Save SMTP Settings
                </button>
                <button type="submit" name="email_action" value="test_smtp" class="btn btn-outline-secondary btn-sm">
                  ✉ Send Test Email
                </button>
              </div>
            </div>
          </form>

          <hr>

          <!-- EMAIL RECIPIENTS -->
          <h6 class="mb-3">Email Recipients</h6>

          <form method="post" class="row g-2 mb-3">
            <input type="hidden" name="email_action" value="add_recipient">
            <div class="col-md-8">
              <input type="email" name="new_recipient" class="form-control form-control-sm"
                     placeholder="recipient@example.com" required>
            </div>
            <div class="col-md-4 d-grid d-md-flex justify-content-md-end">
              <button type="submit" class="btn btn-success btn-sm">
                ➕ Add recipient
              </button>
            </div>
          </form>

          <?php if (empty($emailRecipients['to'])): ?>
            <div class="text-muted small mb-3">No recipients configured yet.</div>
          <?php else: ?>
            <div class="table-responsive mb-3">
              <table class="table table-sm align-middle mb-0">
                <thead>
                  <tr>
                    <th style="width:60%">Email</th>
                    <th style="width:40%" class="text-end">Actions</th>
                  </tr>
                </thead>
                <tbody>
                  <?php foreach ($emailRecipients['to'] as $mail): ?>
                    <tr>
                      <td><?= htmlspecialchars($mail) ?></td>
                      <td class="text-end">
                        <form method="post" class="d-inline">
                          <input type="hidden" name="email_action" value="remove_recipient">
                          <input type="hidden" name="recipient_email" value="<?= htmlspecialchars($mail) ?>">
                          <button type="submit" class="btn btn-outline-danger btn-sm">
                            🗑 Remove
                          </button>
                        </form>
                      </td>
                    </tr>
                  <?php endforeach; ?>
                </tbody>
              </table>
            </div>
          <?php endif; ?>

          <hr>

          <!-- TIMEOUT ALERTS -->
          <h6 class="mb-3">Timeout Alerts (Keyence barcode inactivity)</h6>
          <form method="post">
            <input type="hidden" name="alerts_action" value="save_alerts">

            <div class="row g-3 align-items-center">
              <div class="col-md-4">
                <div class="form-check">
                  <input class="form-check-input" type="checkbox" id="alerts_enabled"
                         name="alerts_enabled" value="1"
                         <?= !empty($alertsCfg['enabled']) ? 'checked' : '' ?>>
                  <label class="form-check-label" for="alerts_enabled">
                    Enable inactivity reports
                  </label>
                </div>
              </div>
              <div class="col-md-4">
                <label class="form-label mb-0">Timeout (minutes)</label>
                <input type="number" min="1" max="1440"
                       name="timeout_minutes"
                       class="form-control form-control-sm"
                       value="<?= htmlspecialchars((string)($alertsCfg['timeout_minutes'] ?? 60)) ?>">
              </div>
              <div class="col-md-4 text-end mt-3 mt-md-0">
                <button type="submit" class="btn btn-primary btn-sm">
                  💾 Save alert settings
                </button>
              </div>
            </div>

            <div class="small text-muted mt-2">
              When enabled, an automatic production report will be emailed if no barcode is read
              for more than the configured number of minutes.
            </div>
          </form>

        </div>
      </div>
    </div>
  </div>

</div>

<?php
// === Keyence & Cloudflare status helpers ===
if (!function_exists('sp_cmd_run')) {
    function sp_cmd_run($cmd) {
        $out = @shell_exec($cmd . ' 2>&1');
        if ($out === null) $out = '';
        return trim($out);
    }
}

// Windows services names
$keyence_service = 'keyence_listener';
$cloud_service   = 'Cloudflared';

// Read current status (before actions)
$keyence_raw = sp_cmd_run('sc query ' . $keyence_service);
$keyence_running = stripos($keyence_raw, 'RUNNING') !== false;
$keyence_stopped = stripos($keyence_raw, 'STOPPED') !== false;

$cloud_raw = sp_cmd_run('sc query ' . $cloud_service);
$cloud_running = stripos($cloud_raw, 'RUNNING') !== false;
$cloud_stopped = stripos($cloud_raw, 'STOPPED') !== false;

// Device info
$keyence_ip   = '192.168.1.180';
$keyence_port = 8600;

// Handle actions (from POST or GET)
$keyence_console = '';
$cloud_console   = '';

$ka = $_POST['keyence_action'] ?? $_GET['keyence_action'] ?? null;
$ca = $_POST['cloud_action']   ?? $_GET['cloud_action']   ?? null;

if ($ka) {
    if ($ka === 'start') {
        $keyence_console = sp_cmd_run('nssm start ' . $keyence_service);
    } elseif ($ka === 'stop') {
        $keyence_console = sp_cmd_run('nssm stop ' . $keyence_service);
    } elseif ($ka === 'restart') {
        $keyence_console = sp_cmd_run('nssm restart ' . $keyence_service);
    } elseif ($ka === 'ping') {
        $keyence_console = sp_cmd_run('ping -n 2 ' . escapeshellarg($keyence_ip));
    } elseif ($ka === 'telnet') {
        $errno = 0; $errstr = '';
        $t0 = microtime(true);
        $fp = @fsockopen($keyence_ip, $keyence_port, $errno, $errstr, 2);
        $dt = (microtime(true) - $t0) * 1000;
        if ($fp) {
            fclose($fp);
            $keyence_console = "TCP OK to {$keyence_ip}:{$keyence_port} ({$dt} ms)";
        } else {
            $keyence_console = "TCP FAIL to {$keyence_ip}:{$keyence_port} — {$errstr} ({$errno})";
        }
    }
}

if ($ca) {
    if ($ca === 'start') {
        $cloud_console = sp_cmd_run('sc start ' . $cloud_service);
    } elseif ($ca === 'stop') {
        $cloud_console = sp_cmd_run('sc stop ' . $cloud_service);
    } elseif ($ca === 'restart') {
        $cloud_console  = sp_cmd_run('sc stop ' . $cloud_service);
        $cloud_console .= "\n" . sp_cmd_run('sc start ' . $cloud_service);
    } elseif ($ca === 'status') {
        $cloud_console = sp_cmd_run('sc query ' . $cloud_service);
    }
}

// quick ping/port status for badges
$keyence_ping_ok = false;
$ping_out = sp_cmd_run('ping -n 1 ' . escapeshellarg($keyence_ip));
if ($ping_out && stripos($ping_out, 'TTL=') !== false) {
    $keyence_ping_ok = true;
}

$keyence_port_ok = false;
$errno = 0; $errstr = '';
$fp = @fsockopen($keyence_ip, $keyence_port, $errno, $errstr, 1);
if ($fp) { $keyence_port_ok = true; fclose($fp); }

// last keyence message
$last_msg = '(no data yet)';
$last_path = __DIR__ . '/../keyence_listener/last_keyence_message.txt';
if (is_file($last_path)) {
    $last_msg = htmlspecialchars(trim(file_get_contents($last_path)));
}
?>

<div class="container-fluid mt-4" style="max-width:1100px;">

  <!-- KEYENCE SETTINGS SECTION -->
  <div class="card mb-4 shadow-sm border-primary mx-auto" id="keyence-section">
    <div class="card-header bg-primary text-white">
      <strong>🛠 Keyence Settings</strong>
    </div>
    <div class="card-body">

      <div class="row mb-3 text-center">
        <div class="col-md-4 mb-2">
          <h6 class="mb-1">Listener service</h6>
          <?php if ($keyence_running): ?>
            <span class="badge bg-success px-3 py-2">🟢 Running</span>
          <?php elseif ($keyence_stopped): ?>
            <span class="badge bg-danger px-3 py-2">🔴 Stopped</span>
          <?php else: ?>
            <span class="badge bg-secondary px-3 py-2">⚪ Unknown</span>
          <?php endif; ?>
          <div class="small text-muted mt-1"><code><?= htmlspecialchars($keyence_service) ?></code></div>
        </div>

        <div class="col-md-4 mb-2">
          <h6 class="mb-1">Ping <?= htmlspecialchars($keyence_ip) ?></h6>
          <?php if ($keyence_ping_ok): ?>
            <span class="badge bg-success px-3 py-2">🟢 Reachable</span>
          <?php else: ?>
            <span class="badge bg-danger px-3 py-2">🔴 No reply</span>
          <?php endif; ?>
        </div>

        <div class="col-md-4 mb-2">
          <h6 class="mb-1">Port <?= (int)$keyence_port ?></h6>
          <?php if ($keyence_port_ok): ?>
            <span class="badge bg-success px-3 py-2">🟢 OPEN</span>
          <?php else: ?>
            <span class="badge bg-danger px-3 py-2">🔴 CLOSED</span>
          <?php endif; ?>
        </div>
      </div>

      <form method="post" class="mb-3 d-flex flex-wrap gap-2">
        <button name="keyence_action" value="start" class="btn btn-success btn-sm">▶ Start listener</button>
        <button name="keyence_action" value="stop" class="btn btn-danger btn-sm">⏹ Stop listener</button>
        <button name="keyence_action" value="restart" class="btn btn-warning btn-sm">♻ Restart listener</button>
        <button name="keyence_action" value="ping" class="btn btn-outline-primary btn-sm">📡 Ping Keyence</button>
        <button name="keyence_action" value="telnet" class="btn btn-outline-secondary btn-sm">🔌 Test port 8600</button>
        <a href="keyence_monitor.php" class="btn btn-outline-dark btn-sm">🖥 Monitor</a>
        <a href="keyence_logs.php" class="btn btn-outline-dark btn-sm">📄 Logs</a>
        <a href="keyence_restart.php" class="btn btn-outline-dark btn-sm">♻ Restart page</a>
      </form>

      <label class="form-label fw-semibold">Keyence console output</label>
      <pre class="bg-dark text-light p-3 rounded" style="min-height:90px; white-space:pre-wrap;"><?=
        $keyence_console ? htmlspecialchars($keyence_console) : 'Ready.'
      ?></pre>

      <label class="form-label fw-semibold mt-3">Last Keyence message</label>
      <pre class="bg-light p-3 rounded" style="min-height:70px;"><?= $last_msg ?></pre>

    </div>
  </div>

  <!-- CLOUDFLARE TUNNEL SECTION -->
  <div class="card mb-5 shadow-sm border-success mx-auto" id="cloudflare-section">
    <div class="card-header bg-success text-white">
      <strong>🌐 Cloudflare Tunnel</strong>
    </div>
    <div class="card-body text-center">

      <div class="mb-3">
        <h6 class="mb-1">Cloudflared service</h6>
        <?php if ($cloud_running): ?>
          <span class="badge bg-success px-3 py-2">🟢 Active</span>
        <?php elseif ($cloud_stopped): ?>
          <span class="badge bg-danger px-3 py-2">🔴 Stopped</span>
        <?php else: ?>
          <span class="badge bg-secondary px-3 py-2">⚪ Unknown</span>
        <?php endif; ?>
        <div class="small text-muted mt-1"><code><?= htmlspecialchars($cloud_service) ?></code></div>
      </div>

      <form method="post" class="mb-3 d-flex flex-wrap gap-2 justify-content-center">
        <button name="cloud_action" value="start" class="btn btn-success btn-sm">▶ Start tunnel</button>
        <button name="cloud_action" value="stop" class="btn btn-danger btn-sm">⏹ Stop tunnel</button>
        <button name="cloud_action" value="restart" class="btn btn-warning btn-sm">♻ Restart tunnel</button>
        <button name="cloud_action" value="status" class="btn btn-outline-primary btn-sm">📊 Status</button>
      </form>

      <label class="form-label fw-semibold">Cloudflared console output</label>
      <pre class="bg-dark text-light p-3 rounded" style="min-height:90px; white-space:pre-wrap;"><?=
        $cloud_console ? htmlspecialchars($cloud_console) : 'Ready.'
      ?></pre>

    </div>
  </div>

</div>

<?php include '../includes/footer.php'; ?>
